
(function ($) {
    'use strict';
    const ALLERGENS_EU = [
        'celery', 'cereals containing gluten', 'crustaceans', 'eggs', 'fish',
        'lupin', 'milk', 'molluscs', 'mustard', 'peanuts', 'sesame',
        'soybeans', 'sulphur dioxide', 'sulphites', 'tree nuts',
        'wheat', 'rye', 'barley', 'oats', 'almonds', 'hazelnuts', 'walnuts',
        'cashews', 'pecans', 'brazil nuts', 'pistachio', 'macadamia'
    ];

    const ALLERGENS_FDA = [
        'milk', 'eggs', 'fish', 'crustacean shellfish', 'tree nuts', 'peanuts',
        'wheat', 'soybeans', 'sesame'
    ];
    const nutritionTemplates = {
        healthy: {
            energy_kcal: '180',
            fat: '3.5',
            carbs: '28',
            sugar: '6.2',
            protein: '12.5',
            fiber: '5.8',
            salt: '0.3',
            saturated_fat: '0.8',
            trans_fat: '0',
            cholesterol: '5',
            sodium: '120',
            potassium: '420',
            added_sugars: '2.0',
            vitamins: 'Vitamin A: 25%, Vitamin C: 35%, Calcium: 15%, Iron: 20%, Vitamin D: 10%, Vitamin B12: 30%',
            ingredients: 'Whole grain oats, almonds, honey, dried cranberries, sunflower seeds, sea salt, natural vanilla extract',
            allergens: 'Contains: Tree nuts (almonds). May contain: Milk, Soy, Wheat',
            barcode: '8901030865278'
        },
        snack: {
            energy_kcal: '485',
            fat: '24.5',
            carbs: '62.3',
            sugar: '32.8',
            protein: '6.2',
            fiber: '2.4',
            salt: '0.5',
            saturated_fat: '14.2',
            trans_fat: '0.3',
            cholesterol: '15',
            sodium: '200',
            potassium: '180',
            added_sugars: '28.5',
            vitamins: 'Vitamin A: 2%, Calcium: 8%, Iron: 10%',
            ingredients: 'Sugar, cocoa butter, whole milk powder, cocoa mass, lecithin (soy), natural vanilla flavoring',
            allergens: 'Contains: Milk, Soy. May contain: Tree nuts, Peanuts, Wheat',
            barcode: '7622210081001'
        },
        beverage: {
            energy_kcal: '42',
            fat: '0',
            carbs: '10.6',
            sugar: '10.6',
            protein: '0',
            fiber: '0',
            salt: '0.01',
            saturated_fat: '0',
            trans_fat: '0',
            cholesterol: '0',
            sodium: '5',
            potassium: '35',
            added_sugars: '0',
            vitamins: 'Vitamin C: 120%, Folate: 10%',
            ingredients: 'Carbonated water, high fructose corn syrup, natural flavors, citric acid, sodium benzoate (preservative)',
            allergens: 'None',
            barcode: '5449000000996'
        }
    };
    $(document).ready(function () {
        if ($('#woonutri_nutrition_data').length === 0 && $('[id^="woonutri_profiles"]').length === 0) {
            return;
        }
        $('.woonutri-quick-fill').on('click', function () {
            const type = $(this).data('type');
            const template = nutritionTemplates[type];

            if (template) {
                fillNutritionData(template);
                showNotification('Sample data filled successfully!', 'success');
            }
        });
        $('.woonutri-clear-all').on('click', function () {
            if (confirm(woonutri_admin.i18n?.clear_confirm || 'Are you sure you want to clear all nutrition data?')) {
                clearAllFields();
                showNotification('All fields cleared!', 'info');
            }
        });
        $(document).on('click', '.woonutri-remove-profile', function (e) {
            e.preventDefault();
            e.stopPropagation();

            const $profileGroup = $(this).closest('.woonutri-profile-group');
            const profileCount = $('.woonutri-profile-group').length;
            if (profileCount <= 1) {
                showNotification('Cannot remove the only profile. Clear fields instead.', 'warning');
                return false;
            }

            if (confirm(woonutri_admin.i18n?.remove_profile_confirm || 'Are you sure you want to remove this nutrition profile?')) {
                $profileGroup.fadeOut(300, function () {
                    $(this).remove();

                    reindexProfiles();
                    showNotification('Profile removed!', 'info');
                });
            }

            return false;
        });
        $('.woonutri-barcode-scan').on('click', function () {
            handleBarcodeScan();
        });
        $('#woonutri_nutrition_data input[type="number"]').on('blur', function () {
            validateNumericField($(this));
        });
        initializeTooltips();

        setupAutoCalculations();
        if (woonutri_admin.is_pro) {
            setupSmartAllergens();
            setupServingSizeConverter();
        }
    });

    
    function fillNutritionData(data) {
        for (const [key, value] of Object.entries(data)) {
            const $field = $('#woonutri_' + key);
            if ($field.length) {
                $field.val(value).trigger('change');
                highlightField($field);
            }
        }
    }

    
    function clearAllFields() {
        $('#woonutri_nutrition_data').find('input[type="number"], input[type="text"], textarea').each(function () {
            $(this).val('').trigger('change');
        });
    }

    
    function reindexProfiles() {
        $('.woonutri-profile-group').each(function (newIndex) {
            $(this).find('input, select, textarea').each(function () {
                const $field = $(this);
                const name = $field.attr('name');
                if (name) {
                    $field.attr('name', name.replace(/woonutri_profiles\[\d+\]/, 'woonutri_profiles[' + newIndex + ']'));
                }
                const id = $field.attr('id');
                if (id) {
                    $field.attr('id', id.replace(/woonutri_profiles\[\d+\]/, 'woonutri_profiles[' + newIndex + ']'));
                }
            });
            $(this).find('label').each(function () {
                const labelFor = $(this).attr('for');
                if (labelFor) {
                    $(this).attr('for', labelFor.replace(/woonutri_profiles\[\d+\]/, 'woonutri_profiles[' + newIndex + ']'));
                }
            });
        });
    }

    
    function highlightField($field) {
        $field.parent().addClass('woonutri-highlight');
        setTimeout(() => {
            $field.parent().removeClass('woonutri-highlight');
        }, 1500);
    }

    
    function validateNumericField($field) {
        const value = $field.val();
        const min = parseFloat($field.attr('min')) || 0;

        if (value !== '' && (isNaN(value) || parseFloat(value) < min)) {
            $field.addClass('woonutri-error');
            showFieldError($field, 'Please enter a valid number');
        } else {
            $field.removeClass('woonutri-error');
            removeFieldError($field);
        }
    }

    
    function showFieldError($field, message) {
        removeFieldError($field);
        $field.after('<span class="woonutri-field-error">' + message + '</span>');
    }

    
    function removeFieldError($field) {
        $field.siblings('.woonutri-field-error').remove();
    }

    
    function initializeTooltips() {

        const tooltips = {
            'woonutri_energy_kcal': 'Total calories per 100g of product',
            'woonutri_saturated_fat': 'Saturated fats should typically be less than 1/3 of total fat',
            'woonutri_added_sugars': 'Sugars added during processing, not naturally occurring',
            'woonutri_sodium': 'Sodium in mg. Note: Salt (g) × 400 ≈ Sodium (mg)',
            'woonutri_barcode': 'EAN-13 or UPC-A barcode for product identification'
        };

        for (const [fieldId, tooltip] of Object.entries(tooltips)) {
            const $field = $('#' + fieldId);
            if ($field.length) {
                $field.attr('title', tooltip);
            }
        }
    }

    
    const FDA_DAILY_VALUES = {
        fat: 78,           // g
        saturated_fat: 20, // g
        trans_fat: 0,      // No DV established
        cholesterol: 300,  // mg
        sodium: 2300,      // mg
        carbs: 275,        // g
        fiber: 28,         // g
        sugar: 50,         // g (added sugars)
        added_sugars: 50,  // g
        protein: 50,       // g
        vitamin_d: 20,     // mcg
        calcium: 1300,     // mg
        iron: 18,          // mg
        potassium: 4700    // mg
    };

    
    const EU_DAILY_VALUES = {
        energy_kcal: 2000, // kcal
        fat: 70,           // g
        saturated_fat: 20, // g
        carbs: 260,        // g
        sugar: 90,         // g (total sugars)
        protein: 50,       // g
        salt: 6,           // g
        fiber: 25,         // g
        vitamin_d: 5,      // mcg
        calcium: 800,      // mg
        iron: 14,          // mg
        potassium: 2000    // mg
    };

    
    function calculateDV(value, nutrientKey) {
        const standard = woonutri_admin.active_standard || 'fda';
        const values = standard === 'eu' ? EU_DAILY_VALUES : FDA_DAILY_VALUES;
        const dv = values[nutrientKey];

        if (!dv || dv === 0 || isNaN(value)) return null;
        return Math.round((parseFloat(value) / dv) * 100);
    }

    
    function setupAutoCalculations() {

        $('#woonutri_salt').on('input', function () {
            const salt = parseFloat($(this).val());
            if (!isNaN(salt) && woonutri_admin.is_pro) {
                const sodium = Math.round(salt * 400); // 1g salt ≈ 400mg sodium
                $('#woonutri_sodium').val(sodium);
                updateDVDisplay('sodium', sodium);
            }
        });
        $('#woonutri_saturated_fat').on('blur', function () {
            const saturatedFat = parseFloat($(this).val());
            const totalFat = parseFloat($('#woonutri_fat').val());

            if (!isNaN(saturatedFat) && !isNaN(totalFat) && saturatedFat > totalFat) {
                showFieldError($(this), 'Saturated fat cannot exceed total fat');
                $(this).val(totalFat);
            }
        });
        $('#woonutri_sugar, #woonutri_added_sugars').on('blur', function () {
            const sugar = parseFloat($('#woonutri_sugar').val()) || 0;
            const addedSugars = parseFloat($('#woonutri_added_sugars').val()) || 0;
            const carbs = parseFloat($('#woonutri_carbs').val()) || 0;

            if (sugar > carbs) {
                showFieldError($('#woonutri_sugar'), 'Sugar cannot exceed total carbohydrates');
                $('#woonutri_sugar').val(carbs);
            }

            if (addedSugars > sugar) {
                showFieldError($('#woonutri_added_sugars'), 'Added sugars cannot exceed total sugar');
                $('#woonutri_added_sugars').val(sugar);
            }
        });
        if (woonutri_admin.is_pro) {
            setupAutoDVCalculation();
        }
    }

    
    function setupSmartAllergens() {
        $('#woonutri_ingredients').on('blur', function () {
            const $field = $(this);
            let text = $field.val();
            if (!text) return;
            const allAllergens = [...new Set([...ALLERGENS_EU, ...ALLERGENS_FDA])];

            allAllergens.forEach(allergen => {
                const regex = new RegExp(`(?<!<strong>)\\b(${allergen})\\b(?!<\\/strong>)`, 'gi');
                text = text.replace(regex, '<strong>$1</strong>');
            });

            if (text !== $field.val()) {
                $field.val(text);
                showNotification('Smart Feature: Allergens automatically bolded!', 'success');
                highlightField($field);
            }
        });
    }

    
    function setupAutoDVCalculation() {
        const dvFields = ['fat', 'saturated_fat', 'cholesterol', 'sodium', 'salt', 'carbs', 'fiber', 'added_sugars', 'sugar', 'protein', 'vitamin_d', 'calcium', 'iron', 'potassium'];

        dvFields.forEach(function (field) {
            const $input = $('#woonutri_' + field);
            if (!$input.length) return;
            addDVIndicator($input, field);
            $input.on('input change', function () {
                updateDVDisplay(field, $(this).val());
            });
            if ($input.val()) {
                updateDVDisplay(field, $input.val());
            }
        });
    }

    
    function addDVIndicator($input, fieldKey) {
        const $wrapper = $input.closest('.woonutri-field-wrapper, .form-field');
        if ($wrapper.find('.woonutri-dv-indicator').length) return;

        const label = (woonutri_admin.active_standard === 'eu') ? '% RI' : '% DV';
        const $indicator = $('<span class="woonutri-dv-indicator" data-field="' + fieldKey + '" title="' + label + ' (based on daily reference)"></span>');
        $input.after($indicator);
    }

    
    function updateDVDisplay(fieldKey, value) {
        const $indicator = $('.woonutri-dv-indicator[data-field="' + fieldKey + '"]');
        if (!$indicator.length) return;

        const dv = calculateDV(value, fieldKey);

        if (dv === null) {
            $indicator.html('').removeClass('woonutri-dv-low woonutri-dv-medium woonutri-dv-high');
            return;
        }
        let badgeType = ''; // Neutral by default
        if (dv >= 20) badgeType = 'warning'; // High
        else if (dv <= 5) badgeType = 'success'; // Low (good)
        const beneficial = ['fiber', 'protein', 'vitamin_d', 'calcium', 'iron', 'potassium'];
        if (beneficial.includes(fieldKey)) {
            if (dv >= 20) badgeType = 'success'; // High (good)
            else if (dv <= 5) badgeType = 'warning'; // Low
        }

        const label = (woonutri_admin.active_standard === 'eu') ? 'RI' : 'DV';
        const badgeClass = badgeType ? `woonutri-badge ${badgeType}` : 'woonutri-badge';

        $indicator.html(`<span class="${badgeClass}">${dv}% ${label}</span>`);
    }

    
    function handleBarcodeScan() {
        if (!woonutri_admin.is_pro) {
            alert('This is a Pro feature. Please upgrade to use barcode scanning.');
            return;
        }

        const barcode = prompt('Enter product barcode:');
        if (barcode) {
            lookupBarcodeData(barcode);
        }
    }

    
    function lookupBarcodeData(barcode) {
        showNotification('Looking up barcode...', 'info');

        $.ajax({
            url: woonutri_admin.ajax_url,
            type: 'POST',
            data: {
                action: 'woonutri_lookup_barcode',
                nonce: woonutri_admin.nonce,
                barcode: barcode
            },
            success: function (response) {
                if (response.success) {
                    const data = response.data;
                    if (data.product_name || data.brand) {
                        const productInfo = [];
                        if (data.brand) productInfo.push(data.brand);
                        if (data.product_name) productInfo.push(data.product_name);
                        showNotification(`Found: ${productInfo.join(' - ')}`, 'success');
                    }
                    fillNutritionData(data);
                    if (!$('#woonutri_barcode').val()) {
                        $('#woonutri_barcode').val(barcode);
                    }
                } else {
                    showNotification(response.data.message || 'Product not found in database', 'warning');
                }
            },
            error: function () {
                showNotification('Error looking up barcode', 'error');
            }
        });
    }

    
    function showNotification(message, type = 'info') {
        const typeClasses = {
            'success': 'notice-success',
            'error': 'notice-error',
            'warning': 'notice-warning',
            'info': 'notice-info'
        };

        const $notice = $('<div class="notice ' + typeClasses[type] + ' is-dismissible woonutri-notice"><p>' + message + '</p></div>');

        $('#woonutri_nutrition_data').prepend($notice);
        setTimeout(() => {
            $notice.fadeOut(() => $notice.remove());
        }, 5000);
        $notice.on('click', '.notice-dismiss', function () {
            $notice.fadeOut(() => $notice.remove());
        });
    }

    
    function setupServingSizeConverter() {

        function injectConverterUI() {

            let $servingSizeWrapper = $('[id="woonutri_profiles[0][serving_size]"]').closest('.woonutri-field-wrapper');
            if ($servingSizeWrapper.length === 0) {
                $servingSizeWrapper = $('#woonutri_serving_size').closest('.woonutri-field-wrapper');
            }

            if ($servingSizeWrapper.length > 0 && $servingSizeWrapper.find('.woonutri-converter-toolbar').length === 0) {
                addConverterUI($servingSizeWrapper);
                return true; // Successfully injected
            }
            return false;
        }
        if (injectConverterUI()) {
            return; // Success, done!
        }
        $(document).on('click', '.woonutri_nutrition_options a, a[href="#woonutri_nutrition_data"]', function () {

            setTimeout(injectConverterUI, 100);
        });
        $(document).on('click', '.product_data_tabs li a', function () {
            const href = $(this).attr('href');
            if (href && href.includes('woonutri')) {
                setTimeout(injectConverterUI, 100);
            }
        });
        let retryCount = 0;
        const maxRetries = 15; // 15 retries * 200ms = 3 seconds max

        function tryInject() {
            if (retryCount >= maxRetries) return;
            retryCount++;

            if (!injectConverterUI()) {
                setTimeout(tryInject, 200);
            }
        }
        setTimeout(tryInject, 300);
        const nutritionPanel = document.getElementById('woonutri_nutrition_data');
        if (nutritionPanel) {
            const observer = new MutationObserver(function (mutations) {
                mutations.forEach(function (mutation) {
                    if (mutation.addedNodes.length > 0) {
                        injectConverterUI();
                    }
                });
            });

            observer.observe(nutritionPanel, {
                childList: true,
                subtree: true
            });
            setTimeout(function () {
                observer.disconnect();
            }, 10000);
        }
    }

    
    function addConverterUI($wrapper) {

        if ($wrapper.find('.woonutri-converter-toolbar').length > 0) return;

        const converterHTML = `
            <div class="woonutri-converter-toolbar" style="margin-top: 10px; padding: 12px; background: linear-gradient(135deg, #f0f7ff 0%, #e8f4fd 100%); border-radius: 8px; border: 1px solid #c3ddf0;">
                <div style="display: flex; align-items: center; gap: 12px; flex-wrap: wrap;">
                    <span style="font-weight: 600; color: #1e3a5f; font-size: 13px;">
                        🔄 Convert Values:
                    </span>
                    <select id="woonutri-convert-direction" class="woonutri-select" style="padding: 6px 10px; border-radius: 4px; border: 1px solid #c3ddf0; background: white; font-size: 12px;">
                        <option value="to_serving">Per 100g → Per Serving</option>
                        <option value="to_100g">Per Serving → Per 100g</option>
                    </select>
                    <button type="button" id="woonutri-convert-btn" class="button button-primary" style="font-size: 12px; padding: 4px 12px;">
                        Convert Now
                    </button>
                    <span class="woonutri-converter-info" style="font-size: 11px; color: #5a7a9a;">
                        ℹ️ Enter serving size with grams (e.g., "30g" or "1 cup (240ml)")
                    </span>
                </div>
            </div>
        `;

        $wrapper.append(converterHTML);
        $('#woonutri-convert-btn').on('click', function () {
            performConversion();
        });
    }

    
    function extractServingSizeGrams(servingSizeText) {
        if (!servingSizeText) return null;
        let match = servingSizeText.match(/(\d+(?:\.\d+)?)\s*g(?:rams?)?/i);
        if (match) return parseFloat(match[1]);
        match = servingSizeText.match(/(\d+(?:\.\d+)?)\s*ml/i);
        if (match) return parseFloat(match[1]);
        const conversions = {
            'tbsp': 15,
            'tablespoon': 15,
            'tsp': 5,
            'teaspoon': 5,
            'cup': 240,
            'oz': 28.35
        };

        for (const [unit, factor] of Object.entries(conversions)) {
            const regex = new RegExp(`(\\d+(?:\\.\\d+)?)\\s*${unit}s?`, 'i');
            match = servingSizeText.match(regex);
            if (match) return parseFloat(match[1]) * factor;
        }
        match = servingSizeText.match(/(\d+(?:\.\d+)?)/);
        if (match) {
            const num = parseFloat(match[1]);
            if (num >= 5 && num <= 500) return num;
        }

        return null;
    }

    
    function performConversion() {

        let servingSizeText = $('[id="woonutri_profiles[0][serving_size]"]').val() ||
            $('#woonutri_serving_size').val();

        if (!servingSizeText) {
            showNotification('Please enter a serving size first (e.g., "30g" or "1 cup (240ml)")', 'warning');
            return;
        }

        const servingSizeG = extractServingSizeGrams(servingSizeText);

        if (!servingSizeG || servingSizeG <= 0) {
            showNotification('Could not determine serving size in grams. Please include grams (e.g., "30g")', 'warning');
            return;
        }

        const direction = $('#woonutri-convert-direction').val();
        const multiplier = direction === 'to_serving' ? (servingSizeG / 100) : (100 / servingSizeG);
        const convertibleFields = [
            'energy_kcal', 'energy_kj', 'fat', 'saturated_fat', 'trans_fat',
            'cholesterol', 'sodium', 'carbs', 'fiber', 'sugar', 'added_sugars',
            'protein', 'salt', 'potassium', 'vitamin_d', 'calcium', 'iron'
        ];

        let convertedCount = 0;

        convertibleFields.forEach(field => {

            let $input = $(`[id="woonutri_profiles[0][${field}]"]`);
            if ($input.length === 0) {
                $input = $(`#woonutri_${field}`);
            }

            if ($input.length && $input.val()) {
                const currentValue = parseFloat($input.val());
                if (!isNaN(currentValue)) {
                    const newValue = Math.round(currentValue * multiplier * 100) / 100;
                    $input.val(newValue).trigger('change');
                    highlightField($input);
                    convertedCount++;
                }
            }
        });

        if (convertedCount > 0) {
            const directionText = direction === 'to_serving'
                ? `per ${servingSizeG}g serving`
                : 'per 100g';
            showNotification(`✅ Converted ${convertedCount} fields to ${directionText}`, 'success');
        } else {
            showNotification('No numeric values found to convert', 'info');
        }
    }

})(jQuery);