import React, { useState, useEffect } from 'react';
import { createRoot } from 'react-dom/client';

// Modern UI Components inspired by shadcn/ui
const Button = ({ children, variant = 'default', size = 'default', className = '', onClick, ...props }) => {
  const baseClasses = 'inline-flex items-center justify-center rounded-md text-sm font-medium ring-offset-white transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-slate-950 focus-visible:ring-offset-2 disabled:pointer-events-none disabled:opacity-50';
  
  const variants = {
    default: 'bg-slate-900 text-slate-50 hover:bg-slate-900/90',
    secondary: 'bg-slate-100 text-slate-900 hover:bg-slate-100/80',
    outline: 'border border-slate-200 bg-white hover:bg-slate-100 hover:text-slate-900',
    ghost: 'hover:bg-slate-100 hover:text-slate-900'
  };
  
  const sizes = {
    default: 'h-10 px-4 py-2',
    sm: 'h-9 rounded-md px-3',
    lg: 'h-11 rounded-md px-8'
  };
  
  return (
    <button
      className={`${baseClasses} ${variants[variant]} ${sizes[size]} ${className}`}
      onClick={onClick}
      {...props}
    >
      {children}
    </button>
  );
};

const Card = ({ children, className = '' }) => (
  <div className={`rounded-lg border border-slate-200 bg-white text-slate-950 shadow-sm ${className}`}>
    {children}
  </div>
);

const CardHeader = ({ children, className = '' }) => (
  <div className={`flex flex-col space-y-1.5 p-6 ${className}`}>
    {children}
  </div>
);

const CardTitle = ({ children, className = '' }) => (
  <h3 className={`text-2xl font-semibold leading-none tracking-tight ${className}`}>
    {children}
  </h3>
);

const CardContent = ({ children, className = '' }) => (
  <div className={`p-6 pt-0 ${className}`}>
    {children}
  </div>
);

const Tabs = ({ children, value, onValueChange, className = '' }) => (
  <div className={`w-full ${className}`} data-active-tab={value}>
    {React.Children.map(children, child =>
      React.cloneElement(child, { activeTab: value, onTabChange: onValueChange })
    )}
  </div>
);

const TabsList = ({ children, activeTab, onTabChange, className = '' }) => (
  <div className={`inline-flex h-10 items-center justify-center rounded-md bg-slate-100 p-1 text-slate-500 ${className}`}>
    {React.Children.map(children, child =>
      React.cloneElement(child, { activeTab, onTabChange })
    )}
  </div>
);

const TabsTrigger = ({ children, value, activeTab, onTabChange, className = '' }) => (
  <button
    className={`inline-flex items-center justify-center whitespace-nowrap rounded-sm px-3 py-1.5 text-sm font-medium ring-offset-white transition-all focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-slate-950 focus-visible:ring-offset-2 disabled:pointer-events-none disabled:opacity-50 ${
      activeTab === value
        ? 'bg-white text-slate-950 shadow-sm'
        : 'hover:bg-white/50 hover:text-slate-900'
    } ${className}`}
    onClick={() => onTabChange(value)}
  >
    {children}
  </button>
);

const TabsContent = ({ children, value, activeTab, className = '' }) => {
  if (activeTab !== value) return null;
  return (
    <div className={`mt-2 ring-offset-white focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-slate-950 focus-visible:ring-offset-2 ${className}`}>
      {children}
    </div>
  );
};

const Input = ({ className = '', ...props }) => (
  <input
    className={`flex h-10 w-full rounded-md border border-slate-200 bg-white px-3 py-2 text-sm ring-offset-white file:border-0 file:bg-transparent file:text-sm file:font-medium placeholder:text-slate-500 focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-slate-950 focus-visible:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50 ${className}`}
    {...props}
  />
);

const Label = ({ children, className = '', ...props }) => (
  <label
    className={`text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70 ${className}`}
    {...props}
  >
    {children}
  </label>
);

const Select = ({ children, value, onValueChange, className = '' }) => (
  <select
    value={value}
    onChange={(e) => onValueChange(e.target.value)}
    className={`flex h-10 w-full rounded-md border border-slate-200 bg-white px-3 py-2 text-sm ring-offset-white focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-slate-950 focus-visible:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50 ${className}`}
  >
    {children}
  </select>
);

const Checkbox = ({ checked, onCheckedChange, className = '', ...props }) => (
  <input
    type="checkbox"
    checked={checked}
    onChange={(e) => onCheckedChange(e.target.checked)}
    className={`h-4 w-4 rounded border-slate-300 text-slate-600 focus:ring-slate-500 ${className}`}
    {...props}
  />
);

// Main WooNutri Settings Component
const WooNutriSettings = () => {
  const [activeTab, setActiveTab] = useState('display');
  const [settings, setSettings] = useState({
    display_position: 'tab',
    tab_title: 'Nutrition Facts',
    tab_priority: '50',
    tab_show_icon: true,
    label_theme: 'basic',
    primary_color: '#2563eb',
    background_color: '#ffffff',
    text_color: '#1f2937',
    font_family: 'default'
  });

  const updateSetting = (key, value) => {
    setSettings(prev => ({ ...prev, [key]: value }));
  };

  return (
    <div className="woonutri-react-admin">
      <div className="max-w-6xl mx-auto p-6 space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-slate-900">WooNutri Settings</h1>
            <p className="text-slate-600 mt-1">Configure how nutrition labels appear on your WooCommerce products.</p>
          </div>
          <Button size="lg">Save Changes</Button>
        </div>

        {/* Main Content */}
        <Tabs value={activeTab} onValueChange={setActiveTab}>
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="display">
              <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
              </svg>
              Display
            </TabsTrigger>
            <TabsTrigger value="appearance">
              <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 21a4 4 0 01-4-4V5a2 2 0 012-2h4a2 2 0 012 2v12a4 4 0 01-4 4zM21 5a2 2 0 00-2-2h-4a2 2 0 00-2 2v12a4 4 0 004 4h4a2 2 0 002-2V5z" />
              </svg>
              Appearance
            </TabsTrigger>
            <TabsTrigger value="advanced">
              <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6V4m0 2a2 2 0 100 4m0-4a2 2 0 110 4m-6 8a2 2 0 100-4m0 4a2 2 0 100 4m0-4v2m0-6V4m6 6v10m6-2a2 2 0 100-4m0 4a2 2 0 100 4m0-4v2m0-6V4" />
              </svg>
              Advanced
            </TabsTrigger>
            <TabsTrigger value="pro">
              <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11.049 2.927c.3-.921 1.603-.921 1.902 0l1.519 4.674a1 1 0 00.95.69h4.915c.969 0 1.371 1.24.588 1.81l-3.976 2.888a1 1 0 00-.363 1.118l1.518 4.674c.3.922-.755 1.688-1.538 1.118l-3.976-2.888a1 1 0 00-1.176 0l-3.976 2.888c-.783.57-1.838-.197-1.538-1.118l1.518-4.674a1 1 0 00-.363-1.118l-3.976-2.888c-.784-.57-.38-1.81.588-1.81h4.914a1 1 0 00.951-.69l1.519-4.674z" />
              </svg>
              Pro Features
            </TabsTrigger>
          </TabsList>

          {/* Display Tab */}
          <TabsContent value="display">
            <Card>
              <CardHeader>
                <CardTitle>Display Position</CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="space-y-4">
                  <Label>Display Location</Label>
                  <div className="space-y-3">
                    <div className="flex items-center space-x-2">
                      <input
                        type="radio"
                        id="summary"
                        name="display_position"
                        value="summary"
                        checked={settings.display_position === 'summary'}
                        onChange={(e) => updateSetting('display_position', e.target.value)}
                        className="h-4 w-4"
                      />
                      <Label htmlFor="summary">Product Summary Only</Label>
                    </div>
                    <div className="flex items-center space-x-2">
                      <input
                        type="radio"
                        id="tab"
                        name="display_position"
                        value="tab"
                        checked={settings.display_position === 'tab'}
                        onChange={(e) => updateSetting('display_position', e.target.value)}
                        className="h-4 w-4"
                      />
                      <Label htmlFor="tab">Product Tab Only</Label>
                    </div>
                    <div className="flex items-center space-x-2">
                      <input
                        type="radio"
                        id="both"
                        name="display_position"
                        value="both"
                        checked={settings.display_position === 'both'}
                        onChange={(e) => updateSetting('display_position', e.target.value)}
                        className="h-4 w-4"
                      />
                      <Label htmlFor="both">Both Summary and Tab</Label>
                    </div>
                  </div>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="tab_title">Tab Title</Label>
                  <Input
                    id="tab_title"
                    value={settings.tab_title}
                    onChange={(e) => updateSetting('tab_title', e.target.value)}
                    placeholder="Nutrition Facts"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="tab_priority">Tab Priority</Label>
                  <Input
                    id="tab_priority"
                    type="number"
                    value={settings.tab_priority}
                    onChange={(e) => updateSetting('tab_priority', e.target.value)}
                    placeholder="50"
                  />
                </div>

                <div className="flex items-center space-x-2">
                  <Checkbox
                    checked={settings.tab_show_icon}
                    onCheckedChange={(checked) => updateSetting('tab_show_icon', checked)}
                  />
                  <Label>Show nutrition icon in tab title</Label>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Appearance Tab */}
          <TabsContent value="appearance">
            <div className="grid gap-6 md:grid-cols-2">
              <Card>
                <CardHeader>
                  <CardTitle>Theme & Colors</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="label_theme">Label Theme</Label>
                    <Select
                      value={settings.label_theme}
                      onValueChange={(value) => updateSetting('label_theme', value)}
                    >
                      <option value="basic">Basic</option>
                      <option value="minimal">Minimal</option>
                      <option value="clean">Clean</option>
                      <option value="fda">FDA Compliant</option>
                    </Select>
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="primary_color">Primary Color</Label>
                    <Input
                      id="primary_color"
                      type="color"
                      value={settings.primary_color}
                      onChange={(e) => updateSetting('primary_color', e.target.value)}
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="background_color">Background Color</Label>
                    <Input
                      id="background_color"
                      type="color"
                      value={settings.background_color}
                      onChange={(e) => updateSetting('background_color', e.target.value)}
                    />
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Typography</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="font_family">Font Family</Label>
                    <Select
                      value={settings.font_family}
                      onValueChange={(value) => updateSetting('font_family', value)}
                    >
                      <option value="default">Default</option>
                      <option value="sans">Sans Serif</option>
                      <option value="serif">Serif</option>
                      <option value="mono">Monospace</option>
                    </Select>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Advanced Tab */}
          <TabsContent value="advanced">
            <Card>
              <CardHeader>
                <CardTitle>Advanced Settings</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-slate-600">Advanced configuration options coming soon...</p>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Pro Features Tab */}
          <TabsContent value="pro">
            <div className="grid gap-6 md:grid-cols-2">
              <Card className="border-orange-200 bg-gradient-to-br from-orange-50 to-orange-100">
                <CardHeader>
                  <CardTitle className="text-orange-800">🔍 AI OCR Scanning</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-orange-700">Upload nutrition label images and automatically extract data using AI-powered OCR technology.</p>
                </CardContent>
              </Card>

              <Card className="border-blue-200 bg-gradient-to-br from-blue-50 to-blue-100">
                <CardHeader>
                  <CardTitle className="text-blue-800">📊 Barcode Lookup</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-blue-700">Instantly lookup nutrition information by scanning product barcodes from comprehensive databases.</p>
                </CardContent>
              </Card>

              <Card className="border-purple-200 bg-gradient-to-br from-purple-50 to-purple-100">
                <CardHeader>
                  <CardTitle className="text-purple-800">🎨 Premium Themes</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-purple-700">Access professional nutrition label themes including FDA, EU, and UK compliance formats.</p>
                </CardContent>
              </Card>

              <Card className="border-green-200 bg-gradient-to-br from-green-50 to-green-100">
                <CardHeader>
                  <CardTitle className="text-green-800">📈 Bulk Operations</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-green-700">Edit nutrition data for multiple products at once with powerful bulk editing tools.</p>
                </CardContent>
              </Card>
            </div>
            
            <div className="mt-6 text-center">
              <Button size="lg" className="bg-gradient-to-r from-orange-500 to-orange-600 hover:from-orange-600 hover:to-orange-700">
                Upgrade to Pro
              </Button>
            </div>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
};

// Initialize React App
document.addEventListener('DOMContentLoaded', function() {
  const reactContainer = document.getElementById('woonutri-react-admin');
  if (reactContainer) {
    const root = createRoot(reactContainer);
    root.render(<WooNutriSettings />);
  }
});