
(function(blocks, element, blockEditor, components, i18n, data) {
    const { registerBlockType } = blocks;
    const { createElement: el, Fragment, useState, useEffect } = element;
    const { InspectorControls, useBlockProps } = blockEditor;
    const { PanelBody, SelectControl, ToggleControl, Placeholder, Spinner } = components;
    const { __ } = i18n;
    const { useSelect } = data;
    const blockData = window.woonutriBlockData || {};
    const blockIcon = el('svg', {
        width: 24,
        height: 24,
        viewBox: '0 0 24 24',
        fill: 'none',
        xmlns: 'http://www.w3.org/2000/svg'
    },
        el('path', {
            d: 'M19 3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V5H19V19Z',
            fill: 'currentColor'
        }),
        el('path', {
            d: 'M7 7H17V9H7V7Z',
            fill: 'currentColor'
        }),
        el('path', {
            d: 'M7 11H17V13H7V11Z',
            fill: 'currentColor'
        }),
        el('path', {
            d: 'M7 15H14V17H7V15Z',
            fill: 'currentColor'
        })
    );
    registerBlockType('woonutri/nutrition-label', {
        title: blockData.i18n?.blockTitle || 'Nutrition Label',
        description: blockData.i18n?.blockDescription || 'Display nutrition facts for a product.',
        category: 'woocommerce',
        icon: blockIcon,
        keywords: [
            'nutrition',
            'facts',
            'label',
            'food',
            'woocommerce'
        ],
        supports: {
            html: false,
            align: ['wide', 'full'],
        },
        attributes: {
            productId: {
                type: 'number',
                default: 0
            },
            theme: {
                type: 'string',
                default: ''
            },
            showTitle: {
                type: 'boolean',
                default: true
            },
            expandable: {
                type: 'boolean',
                default: true
            }
        },

        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { productId, theme, showTitle, expandable } = attributes;
            const blockProps = useBlockProps();

            const [isLoading, setIsLoading] = useState(false);
            const [previewHtml, setPreviewHtml] = useState('');
            const products = blockData.products || [];
            const themes = blockData.themes || [];
            const isPro = blockData.isPro || false;
            useEffect(() => {
                if (productId > 0) {
                    setIsLoading(true);

                    setTimeout(() => {
                        setIsLoading(false);
                    }, 500);
                }
            }, [productId, theme]);
            const inspectorControls = el(InspectorControls, {},
                el(PanelBody, { title: __('Settings', 'woonutri'), initialOpen: true },
                    el(SelectControl, {
                        label: blockData.i18n?.selectProduct || 'Select Product',
                        value: productId,
                        options: products,
                        onChange: function(value) {
                            setAttributes({ productId: parseInt(value, 10) });
                        }
                    }),
                    el(SelectControl, {
                        label: blockData.i18n?.theme || 'Theme',
                        value: theme,
                        options: themes.map(function(t) {
                            return {
                                value: t.value,
                                label: t.label,
                                disabled: t.disabled
                            };
                        }),
                        onChange: function(value) {

                            const selectedTheme = themes.find(t => t.value === value);
                            if (selectedTheme && selectedTheme.pro && !isPro) {
                                return; // Don't allow selecting pro themes
                            }
                            setAttributes({ theme: value });
                        }
                    }),
                    !isPro && el('p', { className: 'woonutri-block-upgrade-notice' },
                        blockData.i18n?.upgradeForThemes || 'Upgrade to Pro for more themes',
                        ' ',
                        el('a', { href: blockData.upgradeUrl, target: '_blank' }, 'Upgrade')
                    ),
                    el(ToggleControl, {
                        label: blockData.i18n?.showTitle || 'Show Title',
                        checked: showTitle,
                        onChange: function(value) {
                            setAttributes({ showTitle: value });
                        }
                    }),
                    el(ToggleControl, {
                        label: blockData.i18n?.expandable || 'Expandable Sections',
                        checked: expandable,
                        onChange: function(value) {
                            setAttributes({ expandable: value });
                        }
                    })
                )
            );
            let blockContent;

            if (productId === 0) {

                blockContent = el(Placeholder, {
                    icon: blockIcon,
                    label: blockData.i18n?.blockTitle || 'Nutrition Label',
                    instructions: blockData.i18n?.noProductSelected || 'Select a product to display its nutrition label.'
                },
                    el(SelectControl, {
                        value: productId,
                        options: products,
                        onChange: function(value) {
                            setAttributes({ productId: parseInt(value, 10) });
                        }
                    })
                );
            } else if (isLoading) {

                blockContent = el('div', { className: 'woonutri-block-loading' },
                    el(Spinner, {}),
                    el('p', {}, __('Loading preview...', 'woonutri'))
                );
            } else {

                const selectedProduct = products.find(p => p.value === productId);
                const productName = selectedProduct ? selectedProduct.label : 'Product';
                const selectedTheme = themes.find(t => t.value === theme);
                const themeName = selectedTheme ? selectedTheme.label : 'Default';

                blockContent = el('div', { className: 'woonutri-block-preview' },
                    el('div', { className: 'woonutri-block-preview-header' },
                        el('span', { className: 'woonutri-block-icon' }, '🥗'),
                        el('span', { className: 'woonutri-block-title' }, blockData.i18n?.blockTitle || 'Nutrition Label')
                    ),
                    el('div', { className: 'woonutri-block-preview-content' },
                        el('div', { className: 'woonutri-block-preview-label' },
                            el('div', { className: 'preview-header' }, 'Nutrition Facts'),
                            el('div', { className: 'preview-row' }),
                            el('div', { className: 'preview-row' }),
                            el('div', { className: 'preview-row' }),
                            el('div', { className: 'preview-row short' })
                        )
                    ),
                    el('div', { className: 'woonutri-block-preview-footer' },
                        el('span', {}, 'Product: ' + productName),
                        el('span', {}, 'Theme: ' + themeName)
                    )
                );
            }

            return el(Fragment, {},
                inspectorControls,
                el('div', blockProps, blockContent)
            );
        },

        save: function() {

            return null;
        }
    });

})(
    window.wp.blocks,
    window.wp.element,
    window.wp.blockEditor,
    window.wp.components,
    window.wp.i18n,
    window.wp.data
);
