
(function($){

    let $preview, $form, fontLoader;
    

    let previousSettings = {};
    let currentTheme = 'basic';
    

    const WOONUTRI_PLUGIN_URL = window.woonutri_data && window.woonutri_data.plugin_url ? window.woonutri_data.plugin_url : '';
    

    window.previewTheme = function(theme) {

        currentTheme = theme;
        const sampleData = {
            energy_kcal: $('#woonutri_sample_energy').val() || '230',
            energy_kj: '963',
            fat: $('#woonutri_sample_fat').val() || '8.0',
            saturated_fat: '1.0',
            trans_fat: '0.0',
            cholesterol: '0',
            sodium: '160',
            carbs: $('#woonutri_sample_carbs').val() || '37.0',
            fiber: $('#woonutri_sample_fiber').val() || '4.0',
            sugar: $('#woonutri_sample_sugar').val() || '12.0',
            added_sugars: '10.0',
            protein: $('#woonutri_sample_protein').val() || '3.0',
            salt: $('#woonutri_sample_salt').val() || '0.4',
            potassium: '240',
            vitamin_d: '2',
            calcium: '260',
            iron: '8',
            serving_size: '2/3 cup (55g)',
            servings_per_container: '8',
            ingredients: 'Mandarin Oranges (37.5%), Light Whipping Cream (Milk), Pears (12.5%), Seedless Grapes (7.6%)',
            allergens: 'Contains: Milk. May contain traces of nuts'
        };
        

        if (window.woonutri_data && window.woonutri_data.is_pro) {
            sampleData.cholesterol = $('#woonutri_sample_cholesterol').val() || '0';
            sampleData.trans_fat = $('#woonutri_sample_trans_fat').val() || '0.0';
            sampleData.added_sugars = $('#woonutri_sample_added_sugars').val() || '10.0';
            sampleData.vitamins = $('#woonutri_sample_vitamins').val() || 'Vitamin C: 20%, Vitamin D: 10%';
            sampleData.ingredients = $('#woonutri_sample_ingredients').val() || sampleData.ingredients;
            sampleData.barcode = $('#woonutri_sample_barcode').val() || '1234567890123';
        }
        

        const appearanceSettings = {
            theme: theme,

            primary_color: $('input[name="woonutri_options[primary_color]"]').val() || '#2a7ae2',
            background_color: $('input[name="woonutri_options[background_color]"]').val() || '#ffffff',
            text_color: $('input[name="woonutri_options[text_color]"]').val() || '#333333',
            header_color: $('input[name="woonutri_options[header_color]"]').val() || '#000000',
            border_color: $('input[name="woonutri_options[border_color]"]').val() || '#e0e0e0',

            font_family: $('select[name="woonutri_options[font_family]"]').val() || 'default',
            custom_font: $('#woonutri_custom_font').val() || '',
            font_size: $('select[name="woonutri_options[font_size]"]').val() || 'normal',
            font_size_custom: $('input[name="woonutri_options[font_size_custom]"]').val() || '16',

            table_layout: $('select[name="woonutri_options[table_layout]"]').val() || 'standard',
            border_radius: $('select[name="woonutri_options[border_radius]"]').val() || 'medium'
        };
        $.ajax({
            url: window.woonutri_data ? window.woonutri_data.ajax_url : ajaxurl,
            type: 'POST',
            data: {
                action: 'woonutri_get_preview',
                nonce: window.woonutri_data ? window.woonutri_data.nonce : '',
                sample_data: sampleData,
                settings: appearanceSettings
            },
            beforeSend: function() {
                $('#woonutri-preview').addClass('woonutri-loading');
            },
            success: function(response) {
                $('#woonutri-preview').removeClass('woonutri-loading');

                if (response.success && response.data.html) {
                    $('#woonutri-preview').html(response.data.html);
                }
            },
            error: function(xhr, status, error) {
                $('#woonutri-preview').removeClass('woonutri-loading');
            }
        });
    };

    
    function getSampleData() {

        const isPro = window.woonutri_data && window.woonutri_data.is_pro;
        

        const data = {
            energy_kcal: $('#woonutri_sample_energy').val(),
            fat: $('#woonutri_sample_fat').val(),
            carbs: $('#woonutri_sample_carbs').val(),
            sugar: $('#woonutri_sample_sugar').val(),
            protein: $('#woonutri_sample_protein').val(),
            fiber: $('#woonutri_sample_fiber').val(),
            salt: $('#woonutri_sample_salt').val()
        };
        

        if (isPro) {
            data.cholesterol = $('#woonutri_toggle_cholesterol').is(':checked') ? $('#woonutri_sample_cholesterol').val() : '';
            data.trans_fat = $('#woonutri_toggle_trans_fat').is(':checked') ? $('#woonutri_sample_trans_fat').val() : '';
            data.added_sugars = $('#woonutri_toggle_added_sugars').is(':checked') ? $('#woonutri_sample_added_sugars').val() : '';
            data.vitamins = $('#woonutri_toggle_vitamins').is(':checked') ? $('#woonutri_sample_vitamins').val() : '';
            data.ingredients = $('#woonutri_toggle_ingredients').is(':checked') ? $('#woonutri_sample_ingredients').val() : '';
            data.barcode = $('#woonutri_toggle_barcode').is(':checked') ? $('#woonutri_sample_barcode').val() : '';
            data.ocr = $('#woonutri_toggle_ocr').is(':checked') ? $('#woonutri_sample_ocr').val() : '';
            data.dv = $('#woonutri_toggle_dv').is(':checked') ? $('#woonutri_sample_dv').val() : '';
            data.serving = $('#woonutri_toggle_serving').is(':checked') ? $('#woonutri_sample_serving').val() : '';
        }
        

        if (data.vitamins) {

            if (data.vitamins.includes(',') && !data.vitamins.includes(':')) {
                const vitamins = data.vitamins.split(',').map(v => v.trim());
                const formattedVitamins = vitamins.map(v => {

                    const percent = Math.floor(Math.random() * 80) + 5;
                    return `${v}: ${percent}%`;
                });
                data.vitamins = formattedVitamins.join(', ');
            }
        }
        

        if (data.ingredients && !data.ingredients.includes(':')) {
            data.ingredients = 'Ingredients: ' + data.ingredients;
        }
        
        return data;
    }
    
    
    function loadCustomFont(fontName) {
        if (!fontName) return Promise.resolve();
        

        if (fontLoader) {
            fontLoader.remove();
        }
        
        return new Promise((resolve, reject) => {

            fontLoader = document.createElement('link');
            fontLoader.href = `https://fonts.googleapis.com/css?family=${encodeURIComponent(fontName)}:400,700&display=swap`;
            fontLoader.rel = 'stylesheet';
            

            fontLoader.onload = () => {

                setTimeout(resolve, 100);
            };
            
            fontLoader.onerror = () => {

                resolve();
            };
            
            document.head.appendChild(fontLoader);
        });
    }
    
    
    function getSettings() {

        const isPro = $('#woonutri_label_theme').length > 0;
        

        const selectors = {
            theme: '#woonutri_label_theme',
            color: 'input[name="woonutri_options[primary_color]"]',
            font: 'select[name="woonutri_options[font_family]"]',
            customFont: '#woonutri_custom_font',
            fontSize: 'select[name="woonutri_options[font_size]"]',
            fontSizeCustom: 'input[name="woonutri_options[font_size_custom]"]',
            layout: 'select[name="woonutri_options[table_layout]"]',
            radius: 'select[name="woonutri_options[border_radius]"]',
            showLogo: 'input[name="woonutri_options[show_logo]"]',
            bg: 'input[name="woonutri_options[background_color]"]',
            text: 'input[name="woonutri_options[text_color]"]',
            header: 'input[name="woonutri_options[header_color]"]',
            border: 'input[name="woonutri_options[border_color]"]'
        };
        

        const elementCheck = {};
        const settings = {};
        
        for (const [key, selector] of Object.entries(selectors)) {
            const element = $(selector);
            elementCheck[key] = {
                selector: selector,
                exists: element.length > 0,
                value: element.length > 0 ? (key === 'showLogo' ? element.is(':checked') : element.val()) : null
            };
            
            if (key === 'theme' && !isPro) {

                settings[key] = 'basic';
            } else if (key === 'showLogo') {
                settings[key] = element.is(':checked');
            } else {
                settings[key] = element.val();
            }
        }
        

        const unitSelectors = {
            energy_kcal: 'input[name="woonutri_options[unit_energy_kcal]"]',
            fat: 'input[name="woonutri_options[unit_fat]"]',
            sugar: 'input[name="woonutri_options[unit_sugar]"]',
            protein: 'input[name="woonutri_options[unit_protein]"]',
            fiber: 'input[name="woonutri_options[unit_fiber]"]',
            salt: 'input[name="woonutri_options[unit_salt]"]'
        };
        
        settings.units = {};
        const unitCheck = {};
        
        for (const [key, selector] of Object.entries(unitSelectors)) {
            const element = $(selector);
            unitCheck[key] = {
                selector: selector,
                exists: element.length > 0,
                value: element.length > 0 ? element.val() : null
            };
            settings.units[key] = element.val();
        }

        return settings;
    }
    
    
    function getChangedSettings(newSettings) {
        if (!Object.keys(previousSettings).length) {
            return { hasChanges: true, changes: ['initial'] };
        }
        
        const changes = [];
        

        for (const key in newSettings) {
            if (key === 'units') continue; // Handle units separately
            
            if (JSON.stringify(newSettings[key]) !== JSON.stringify(previousSettings[key])) {
                changes.push(key);
            }
        }
        

        for (const unit in newSettings.units) {
            if (newSettings.units[unit] !== previousSettings.units?.[unit]) {
                changes.push(`unit_${unit}`);
            }
        }
        
        return {
            hasChanges: changes.length > 0,
            changes: changes
        };
    }
    
    
    function highlightChanges(changes) {

        $('.woonutri-highlight-change').removeClass('woonutri-highlight-change');
        

        changes.forEach(change => {
            if (change === 'initial') return;
            
            if (change === 'customFont') {
                $('#woonutri_custom_font').addClass('woonutri-highlight-change');
            } else if (change === 'showLogo') {
                $('input[name="woonutri_options[show_logo]"]').parent().addClass('woonutri-highlight-change');
            } else if (change.startsWith('unit_')) {
                const unit = change.replace('unit_', '');
                $(`input[name="woonutri_options[unit_${unit}]"]`).addClass('woonutri-highlight-change');
            } else {

                $(`#woonutri_label_${change}, input[name="woonutri_options[${change}]"], select[name="woonutri_options[${change}]"]`)
                    .addClass('woonutri-highlight-change');
            }
        });
        

        setTimeout(() => {
            $('.woonutri-highlight-change').removeClass('woonutri-highlight-change');
        }, 1500);
    }
    
    
    async function updatePreview() {
        window.previewTheme(currentTheme);
    }
    

    function debounce(func, wait) {
        let timeout;
        return function() {
            const context = this, args = arguments;
            clearTimeout(timeout);
            timeout = setTimeout(() => {
                func.apply(context, args);
            }, wait);
        };
    }
    

    const debouncedUpdate = debounce(updatePreview, 300);
    

    function addFeedbackStyles() {
        const style = `
            <style id="woonutri-preview-feedback-styles">
                .woonutri-highlight-change {
                    transition: background-color 0.3s ease;
                    background-color: rgba(42, 122, 226, 0.1) !important;
                    box-shadow: 0 0 0 2px rgba(42, 122, 226, 0.3);
                }
                #woonutri-preview .woonutri-label {
                    transition: opacity 0.3s ease;
                }
                .woonutri-settings-changed {
                    position: relative;
                }
                .woonutri-settings-changed:after {
                    content: '✓';
                    position: absolute;
                    right: 10px;
                    top: 50%;
                    transform: translateY(-50%);
                    color: #4CAF50;
                    font-weight: bold;
                    opacity: 1;
                    transition: opacity 0.5s ease;
                    animation: fadeInOut 1.5s ease;
                }
                @keyframes fadeInOut {
                    0% { opacity: 0; }
                    20% { opacity: 1; }
                    80% { opacity: 1; }
                    100% { opacity: 0; }
                }
            </style>
        `;
        
        if (!$('#woonutri-preview-feedback-styles').length) {
            $('head').append(style);
        }
    }
    

    function generateRealisticSample() {
        return {
            energy_kcal: Math.floor(Math.random() * 300) + 100,
            fat: (Math.random() * 20).toFixed(1),
            sugar: (Math.random() * 15).toFixed(1),
            protein: (Math.random() * 25).toFixed(1),
            fiber: (Math.random() * 8).toFixed(1),
            salt: (Math.random() * 2).toFixed(2),
            cholesterol: Math.floor(Math.random() * 100),
            trans_fat: (Math.random() * 1).toFixed(1),
            added_sugars: (Math.random() * 10).toFixed(1),
            vitamins: 'Vitamin A: 15%, Vitamin C: 45%, Calcium: 20%, Iron: 10%',
            ingredients: 'Ingredients: Whole Grain Wheat, Sugar, Corn Syrup, Honey, Salt, Malt Extract, Vitamins & Minerals',
            barcode: '8' + Math.floor(Math.random() * 10000000000000),
            ocr: 'Auto-extracted from packaging',
            dv: Math.floor(Math.random() * 30) + 5,
            serving: (Math.random() * 50 + 20).toFixed(0) + 'g'
        };
    }
    

    function bindEventHandlers() {
        const settingsSelectors = '#woonutri_label_theme, input[name^="woonutri_options"], select[name^="woonutri_options"]';
        const previewSelectors = '#woonutri-preview-controls input, #woonutri-preview-controls select';
        $(document).off('change input', '.woonutri-settings-handler');
        $(document).on('change input', settingsSelectors, function() {
            $(this).addClass('woonutri-settings-changed');
            setTimeout(() => {
                $(this).removeClass('woonutri-settings-changed');
            }, 1500);
            debouncedUpdate();
        }).addClass('woonutri-settings-handler');
        $(document).off('change input', '.woonutri-preview-handler');
        $(document).on('change input', previewSelectors, function() {
            debouncedUpdate();
        }).addClass('woonutri-preview-handler');
    }
    

    $(document).ready(function(){

        addFeedbackStyles();
        $form = $('#woonutri-settings-form');
        $preview = $('#woonutri-preview .woonutri-label');
        window.woonutri_label_preview = window.woonutri_label_preview || {};
        window.woonutri_label_preview.logo_url = $('.woonutri-label-logo img').attr('src') || '';
        

        if (!window.woonutri_label_preview.logo_url) {
            window.woonutri_label_preview.logo_url = WOONUTRI_PLUGIN_URL + 'assets/WooNutri_logo.png';
        }
        bindEventHandlers();
        setTimeout(() => {
            updatePreview();
        }, 100);
        

        $('#woonutri-reset-btn').on('click', function(){

            const defaultColors = {
                'primary_color': '#2a7ae2',
                'background_color': '#ffffff',
                'text_color': '#333333',
                'header_color': '#000000',
                'border_color': '#e0e0e0'
            };
            $.each(defaultColors, function(name, defaultValue) {
                var $input = $('input[name="woonutri_options[' + name + ']"]');
                if ($input.length && typeof $.fn.wpColorPicker === 'function') {
                    $input.val(defaultValue);
                    $input.wpColorPicker('color', defaultValue);
                }
            });

            $('input[name="woonutri_options[display_position]"]').prop('checked', false);
            $('input[name="woonutri_options[display_position]"][value="summary"]').prop('checked', true);

            $('.woonutri-radio-card').removeClass('checked');
            $('input[name="woonutri_options[display_position]"][value="summary"]').closest('.woonutri-radio-card').addClass('checked');

            $('.woonutri-tab-config').slideUp(300);
            $('input[name="woonutri_options[tab_title]"]').val('Nutrition Facts');
            $('input[name="woonutri_options[tab_priority]"]').val('50');
            $('input[name="woonutri_options[tab_show_icon]"]').prop('checked', true);
            $('select[name="woonutri_options[font_family]"]').val('default');
            $('select[name="woonutri_options[font_size]"]').val('normal');
            $('input[name="woonutri_options[custom_font]"]').val('');

            $('.woonutri-custom-font-size').slideUp(200);
            $('select[name="woonutri_options[table_layout]"]').val('standard');
            $('select[name="woonutri_options[border_radius]"]').val('medium');
            $('input[name="woonutri_options[show_logo]"]').prop('checked', false);
            $('input[name="woonutri_options[enable_collapsible]"]').prop('checked', true);
            $('input[name="woonutri_options[show_pdf_button]"]').prop('checked', false);
            $('input[name="woonutri_options[show_nutri_score]"]').prop('checked', true);
            $('input[name="woonutri_options[enable_schema_seo]"]').prop('checked', true);
            $('input[name="woonutri_options[show_traffic_lights]"]').prop('checked', false);
            $('input[name="woonutri_options[display_format]"]').prop('checked', false);
            $('input[name="woonutri_options[display_format]"][value="per_100g"]').prop('checked', true);

            $('input[name="woonutri_options[display_format]"]').closest('.woonutri-radio-card').removeClass('checked');
            $('input[name="woonutri_options[display_format]"][value="per_100g"]').closest('.woonutri-radio-card').addClass('checked');
            const defaultUnits = {
                'energy_kcal': 'kcal',
                'fat': 'g',
                'saturated_fat': 'g',
                'trans_fat': 'g',
                'carbs': 'g',
                'sugar': 'g',
                'added_sugars': 'g',
                'protein': 'g',
                'fiber': 'g',
                'salt': 'g',
                'sodium': 'mg',
                'cholesterol': 'mg',
                'potassium': 'mg',
                'vitamin_d': 'mcg',
                'calcium': 'mg',
                'iron': 'mg'
            };
            $.each(defaultUnits, function(name, defaultValue) {
                $('input[name="woonutri_options[unit_' + name + ']"]').val(defaultValue);
            });
            $('input[name="woonutri_options[label_title]"]').val('');
            $('input[name="woonutri_options[label_per_serving]"]').val('');
            $('input[name="woonutri_options[label_per_100g]"]').val('');
            $('input[name="woonutri_options[label_daily_value]"]').val('');
            $('input[name="woonutri_options[enable_shadow]"]').prop('checked', false);
            $('select[name="woonutri_options[shadow_intensity]"]').val('medium');
            $('input[name="woonutri_options[enable_gradient]"]').prop('checked', false);
            $('input[name="woonutri_options[enable_animations]"]').prop('checked', false);
            $('textarea[name="woonutri_options[custom_css]"]').val('');
            $('#woonutri_sample_energy').val('250');
            $('#woonutri_sample_fat').val('12.5');
            $('#woonutri_sample_carbs').val('45.0');
            $('#woonutri_sample_sugar').val('15.2');
            $('#woonutri_sample_protein').val('8.7');
            $('#woonutri_sample_fiber').val('3.2');
            $('#woonutri_sample_salt').val('0.8');
            $('#woonutri_sample_cholesterol').val('10');
            $('#woonutri_sample_trans_fat').val('0.2');
            $('#woonutri_sample_added_sugars').val('1.5');
            $('#woonutri_sample_vitamins').val('Vit C, D');
            $('#woonutri_sample_ingredients').val('Whey, Cocoa');
            $('#woonutri_sample_barcode').val('1234567890123');
            $('#woonutri_sample_ocr').val('Auto-extracted');
            $('#woonutri_sample_dv').val('15');
            $('#woonutri_sample_serving').val('30g');
            $('#woonutri-preview-controls input[type=checkbox]').prop('checked', true);

            $('.woonutri-theme-btn').removeClass('active');
            $('.woonutri-theme-btn[data-theme="basic"]').addClass('active');
            currentTheme = 'basic';
            updatePreview();
            const $resetMsg = $('<div class="woonutri-reset-message">All settings reset to defaults</div>')
                .css({
                    position: 'fixed',
                    top: '20px',
                    right: '20px',
                    background: '#4CAF50',
                    color: 'white',
                    padding: '10px 15px',
                    borderRadius: '4px',
                    boxShadow: '0 2px 8px rgba(0,0,0,0.2)',
                    zIndex: 9999,
                    opacity: 0
                })
                .appendTo('body')
                .animate({ opacity: 1 }, 300);

            setTimeout(() => {
                $resetMsg.animate({ opacity: 0 }, 300, function() {
                    $(this).remove();
                });
            }, 2000);
        });
        

        if (!$('#woonutri-random-sample-btn').length) {
            const $randomBtn = $('<button type="button" id="woonutri-random-sample-btn" class="button button-secondary">Generate Sample Data</button>')
                .css({
                    marginLeft: '10px'
                });
                
            $('#woonutri-reset-btn').after($randomBtn);
            
            $randomBtn.on('click', function() {
                const sample = generateRealisticSample();
                

                $('#woonutri_sample_energy').val(sample.energy_kcal);
                $('#woonutri_sample_fat').val(sample.fat);
                $('#woonutri_sample_carbs').val(sample.carbs || (Math.random() * 50 + 20).toFixed(1));
                $('#woonutri_sample_sugar').val(sample.sugar);
                $('#woonutri_sample_protein').val(sample.protein);
                $('#woonutri_sample_fiber').val(sample.fiber);
                $('#woonutri_sample_salt').val(sample.salt);
                $('#woonutri_sample_cholesterol').val(sample.cholesterol);
                $('#woonutri_sample_trans_fat').val(sample.trans_fat);
                $('#woonutri_sample_added_sugars').val(sample.added_sugars);
                $('#woonutri_sample_vitamins').val(sample.vitamins);
                $('#woonutri_sample_ingredients').val(sample.ingredients);
                $('#woonutri_sample_barcode').val(sample.barcode);
                $('#woonutri_sample_ocr').val(sample.ocr);
                $('#woonutri_sample_dv').val(sample.dv);
                $('#woonutri_sample_serving').val(sample.serving);
                

                $('#woonutri-preview-controls input[type=checkbox]').prop('checked', true);
                

                updatePreview();
                

                const $msg = $('<div class="woonutri-sample-message">Sample data generated</div>')
                    .css({
                        position: 'fixed',
                        top: '20px',
                        right: '20px',
                        background: '#2196F3',
                        color: 'white',
                        padding: '10px 15px',
                        borderRadius: '4px',
                        boxShadow: '0 2px 8px rgba(0,0,0,0.2)',
                        zIndex: 9999,
                        opacity: 0
                    })
                    .appendTo('body')
                    .animate({ opacity: 1 }, 300);
                    
                setTimeout(() => {
                    $msg.animate({ opacity: 0 }, 300, function() {
                        $(this).remove();
                    });
                }, 2000);
            });
        }
        $(document).on('click', '.woonutri-theme-btn', function(e) {
            e.preventDefault();
            const $btn = $(this);
            const theme = $btn.data('theme');
            $('.woonutri-theme-btn').removeClass('active');
            $btn.addClass('active');
            $('.woonutri-theme-chip').removeClass('active');
            $('.woonutri-theme-chip[data-theme="' + theme + '"]').addClass('active');
            $('#woonutri-theme-input').val(theme);
            window.previewTheme(theme);
        });
    });
})(jQuery);