<?php
/**
 * @package WooNutri
 * @since 1.2.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Woonutri_Bulk_Operations {

    public static function init() {
        add_action( 'admin_menu', [ __CLASS__, 'add_bulk_page' ] );
        add_action( 'admin_enqueue_scripts', [ __CLASS__, 'enqueue_assets' ] );
        add_action( 'wp_ajax_woonutri_bulk_save', [ __CLASS__, 'ajax_bulk_save' ] );
        add_action( 'wp_ajax_woonutri_bulk_export', [ __CLASS__, 'ajax_bulk_export' ] );
        add_action( 'wp_ajax_woonutri_bulk_import', [ __CLASS__, 'ajax_bulk_import' ] );
        add_action( 'wp_ajax_woonutri_get_products_data', [ __CLASS__, 'ajax_get_products_data' ] );
        add_action( 'wp_ajax_woonutri_apply_template', [ __CLASS__, 'ajax_apply_template' ] );
    }

    public static function add_bulk_page() {
        add_submenu_page(
            'woonutri-dashboard',
            __( 'Bulk Editor', 'woonutri' ),
            __( 'Bulk Editor', 'woonutri' ),
            'manage_woocommerce',
            'woonutri-bulk',
            [ __CLASS__, 'render_bulk_page' ]
        );
    }

    public static function enqueue_assets( $hook ) {
        if ( strpos( $hook, 'woonutri-bulk' ) === false ) {
            return;
        }

        wp_enqueue_style(
            'woonutri-bulk',
            WOONUTRI_PLUGIN_URL . 'assets/css/bulk-editor.css',
            [],
            WOONUTRI_VERSION
        );

        wp_enqueue_script(
            'woonutri-bulk',
            WOONUTRI_PLUGIN_URL . 'assets/js/bulk-editor.js',
            [ 'jquery' ],
            WOONUTRI_VERSION,
            true
        );

        $core_fields = [
            'energy_kcal', 'energy_kj', 'fat', 'saturated_fat', 'trans_fat',
            'cholesterol', 'sodium', 'carbs', 'fiber', 'sugar', 'added_sugars',
            'protein', 'salt', 'vitamin_d', 'calcium', 'iron', 'potassium',
            'serving_size', 'servings_per_container'
        ];

        wp_localize_script( 'woonutri-bulk', 'woonutriBulk', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'woonutri_bulk' ),
            'fields' => $core_fields,
            'field_labels' => self::get_field_labels(),
            'templates' => self::get_templates(),
            'i18n' => [
                'saving' => __( 'Saving...', 'woonutri' ),
                'saved' => __( 'Saved!', 'woonutri' ),
                'error' => __( 'Error saving data', 'woonutri' ),
                'confirm_template' => __( 'Apply template to selected products? This will overwrite existing data.', 'woonutri' ),
                'no_products' => __( 'No products selected', 'woonutri' ),
                'export_success' => __( 'Export complete', 'woonutri' ),
                'import_success' => __( 'Import complete', 'woonutri' ),
            ],
        ] );
    }

    public static function get_field_labels() {
        $hierarchy = Woonutri_Meta_Handler::get_field_hierarchy();
        $labels = [];
        foreach ( $hierarchy as $key => $config ) {
            $labels[ $key ] = $config['label'];
        }
        return $labels;
    }

    public static function get_templates() {
        $default_templates = [
            'blank' => [
                'name' => __( 'Blank Template', 'woonutri' ),
                'description' => __( 'Clear all nutrition data', 'woonutri' ),
                'data' => [],
            ],
            'cereal' => [
                'name' => __( 'Breakfast Cereal', 'woonutri' ),
                'description' => __( 'Typical values for breakfast cereals', 'woonutri' ),
                'data' => [
                    'serving_size' => '30g',
                    'energy_kcal' => '110',
                    'energy_kj' => '460',
                    'fat' => '1',
                    'saturated_fat' => '0.2',
                    'carbs' => '24',
                    'sugar' => '8',
                    'fiber' => '3',
                    'protein' => '2',
                    'salt' => '0.3',
                    'sodium' => '120',
                ],
            ],
            'milk' => [
                'name' => __( 'Whole Milk', 'woonutri' ),
                'description' => __( 'Standard whole milk values per 100ml', 'woonutri' ),
                'data' => [
                    'serving_size' => '250ml',
                    'energy_kcal' => '64',
                    'energy_kj' => '268',
                    'fat' => '3.6',
                    'saturated_fat' => '2.3',
                    'carbs' => '4.7',
                    'sugar' => '4.7',
                    'protein' => '3.3',
                    'calcium' => '120',
                    'salt' => '0.1',
                ],
            ],
            'bread' => [
                'name' => __( 'White Bread', 'woonutri' ),
                'description' => __( 'Typical white bread values per 100g', 'woonutri' ),
                'data' => [
                    'serving_size' => '40g (1 slice)',
                    'energy_kcal' => '265',
                    'energy_kj' => '1110',
                    'fat' => '3.2',
                    'saturated_fat' => '0.7',
                    'carbs' => '49',
                    'sugar' => '5',
                    'fiber' => '2.7',
                    'protein' => '9',
                    'salt' => '1',
                    'sodium' => '400',
                ],
            ],
            'yogurt' => [
                'name' => __( 'Plain Yogurt', 'woonutri' ),
                'description' => __( 'Plain yogurt values per 100g', 'woonutri' ),
                'data' => [
                    'serving_size' => '150g',
                    'energy_kcal' => '61',
                    'energy_kj' => '255',
                    'fat' => '3.3',
                    'saturated_fat' => '2.1',
                    'carbs' => '4.7',
                    'sugar' => '4.7',
                    'protein' => '3.5',
                    'calcium' => '121',
                    'salt' => '0.08',
                ],
            ],
            'chicken_breast' => [
                'name' => __( 'Chicken Breast', 'woonutri' ),
                'description' => __( 'Cooked chicken breast per 100g', 'woonutri' ),
                'data' => [
                    'serving_size' => '150g',
                    'energy_kcal' => '165',
                    'energy_kj' => '690',
                    'fat' => '3.6',
                    'saturated_fat' => '1',
                    'carbs' => '0',
                    'sugar' => '0',
                    'protein' => '31',
                    'salt' => '0.1',
                    'sodium' => '74',
                    'cholesterol' => '85',
                ],
            ],
            'apple' => [
                'name' => __( 'Apple (Fresh)', 'woonutri' ),
                'description' => __( 'Raw apple per 100g', 'woonutri' ),
                'data' => [
                    'serving_size' => '182g (1 medium)',
                    'energy_kcal' => '52',
                    'energy_kj' => '218',
                    'fat' => '0.2',
                    'saturated_fat' => '0',
                    'carbs' => '14',
                    'sugar' => '10',
                    'fiber' => '2.4',
                    'protein' => '0.3',
                    'vitamin_c' => '4.6',
                    'potassium' => '107',
                ],
            ],
            'rice' => [
                'name' => __( 'White Rice (Cooked)', 'woonutri' ),
                'description' => __( 'Cooked white rice per 100g', 'woonutri' ),
                'data' => [
                    'serving_size' => '158g (1 cup)',
                    'energy_kcal' => '130',
                    'energy_kj' => '544',
                    'fat' => '0.3',
                    'saturated_fat' => '0.1',
                    'carbs' => '28',
                    'sugar' => '0',
                    'fiber' => '0.4',
                    'protein' => '2.7',
                    'iron' => '0.2',
                ],
            ],
        ];

        $custom_templates = get_option( 'woonutri_custom_templates', [] );

        return array_merge( $default_templates, $custom_templates );
    }

    public static function render_bulk_page() {
        $is_pro = function_exists( 'woonutri_fs' ) && woonutri_fs()->can_use_premium_code();

        if ( ! $is_pro ) {
            self::render_pro_upsell();
            return;
        }

        $categories = get_terms( [
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
        ] );

        $core_fields = [
            'serving_size' => __( 'Serving Size', 'woonutri' ),
            'servings_per_container' => __( 'Servings/Container', 'woonutri' ),
            'energy_kcal' => __( 'Calories (kcal)', 'woonutri' ),
            'energy_kj' => __( 'Energy (kJ)', 'woonutri' ),
            'fat' => __( 'Fat (g)', 'woonutri' ),
            'saturated_fat' => __( 'Sat. Fat (g)', 'woonutri' ),
            'trans_fat' => __( 'Trans Fat (g)', 'woonutri' ),
            'cholesterol' => __( 'Cholesterol (mg)', 'woonutri' ),
            'sodium' => __( 'Sodium (mg)', 'woonutri' ),
            'carbs' => __( 'Carbs (g)', 'woonutri' ),
            'fiber' => __( 'Fiber (g)', 'woonutri' ),
            'sugar' => __( 'Sugars (g)', 'woonutri' ),
            'added_sugars' => __( 'Added Sugars (g)', 'woonutri' ),
            'protein' => __( 'Protein (g)', 'woonutri' ),
            'salt' => __( 'Salt (g)', 'woonutri' ),
            'vitamin_d' => __( 'Vit D (mcg)', 'woonutri' ),
            'calcium' => __( 'Calcium (mg)', 'woonutri' ),
            'iron' => __( 'Iron (mg)', 'woonutri' ),
            'potassium' => __( 'Potassium (mg)', 'woonutri' ),
        ];
        ?>
        <div class="wrap woonutri-bulk-wrap">
            <div class="woonutri-bulk-header">
                <h1><?php _e( 'Bulk Nutrition Editor', 'woonutri' ); ?></h1>
                <div class="bulk-actions-bar">
                    <button type="button" class="button" id="woonutri-select-all">
                        <?php _e( 'Select All', 'woonutri' ); ?>
                    </button>
                    <button type="button" class="button" id="woonutri-deselect-all">
                        <?php _e( 'Deselect All', 'woonutri' ); ?>
                    </button>
                    <select id="woonutri-template-select">
                        <option value=""><?php _e( '-- Apply Template --', 'woonutri' ); ?></option>
                        <?php foreach ( self::get_templates() as $key => $template ) : ?>
                        <option value="<?php echo esc_attr( $key ); ?>"><?php echo esc_html( $template['name'] ); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <button type="button" class="button" id="woonutri-apply-template">
                        <?php _e( 'Apply', 'woonutri' ); ?>
                    </button>
                    <span class="bulk-separator">|</span>
                    <button type="button" class="button" id="woonutri-export-csv">
                        <span class="dashicons dashicons-download"></span>
                        <?php _e( 'Export CSV', 'woonutri' ); ?>
                    </button>
                    <button type="button" class="button" id="woonutri-import-csv">
                        <span class="dashicons dashicons-upload"></span>
                        <?php _e( 'Import CSV', 'woonutri' ); ?>
                    </button>
                    <input type="file" id="woonutri-csv-file" accept=".csv" style="display:none;">
                </div>
            </div>

            <div class="woonutri-bulk-filters">
                <label>
                    <?php _e( 'Category:', 'woonutri' ); ?>
                    <select id="woonutri-filter-category">
                        <option value=""><?php _e( 'All Categories', 'woonutri' ); ?></option>
                        <?php if ( ! is_wp_error( $categories ) ) : ?>
                            <?php foreach ( $categories as $cat ) : ?>
                            <option value="<?php echo esc_attr( $cat->term_id ); ?>">
                                <?php echo esc_html( $cat->name ); ?> (<?php echo $cat->count; ?>)
                            </option>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </select>
                </label>
                <label>
                    <?php _e( 'Status:', 'woonutri' ); ?>
                    <select id="woonutri-filter-status">
                        <option value=""><?php _e( 'All', 'woonutri' ); ?></option>
                        <option value="with"><?php _e( 'With Nutrition Data', 'woonutri' ); ?></option>
                        <option value="without"><?php _e( 'Without Nutrition Data', 'woonutri' ); ?></option>
                    </select>
                </label>
                <label>
                    <?php _e( 'Search:', 'woonutri' ); ?>
                    <input type="text" id="woonutri-filter-search" placeholder="<?php _e( 'Product name...', 'woonutri' ); ?>">
                </label>
                <button type="button" class="button button-primary" id="woonutri-load-products">
                    <?php _e( 'Load Products', 'woonutri' ); ?>
                </button>
            </div>

            <div class="woonutri-bulk-table-container">
                <table class="woonutri-bulk-table" id="woonutri-bulk-table">
                    <thead>
                        <tr>
                            <th class="col-select"><input type="checkbox" id="woonutri-check-all"></th>
                            <th class="col-product"><?php _e( 'Product', 'woonutri' ); ?></th>
                            <?php foreach ( $core_fields as $key => $label ) : ?>
                            <th class="col-field" data-field="<?php echo esc_attr( $key ); ?>"><?php echo esc_html( $label ); ?></th>
                            <?php endforeach; ?>
                            <th class="col-actions"><?php _e( 'Actions', 'woonutri' ); ?></th>
                        </tr>
                    </thead>
                    <tbody id="woonutri-bulk-tbody">
                        <tr class="no-data">
                            <td colspan="<?php echo count( $core_fields ) + 3; ?>">
                                <?php _e( 'Click "Load Products" to start editing', 'woonutri' ); ?>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <div class="woonutri-bulk-footer">
                <button type="button" class="button button-primary button-hero" id="woonutri-save-all">
                    <span class="dashicons dashicons-saved"></span>
                    <?php _e( 'Save All Changes', 'woonutri' ); ?>
                </button>
                <span class="save-status" id="woonutri-save-status"></span>
            </div>
        </div>
        <?php
    }

    private static function render_pro_upsell() {
        ?>
        <div class="wrap woonutri-bulk-wrap">
            <div class="woonutri-pro-upsell-card">
                <div class="upsell-icon">📦</div>
                <h2><?php _e( 'Bulk Editor - Pro Feature', 'woonutri' ); ?></h2>
                <p><?php _e( 'Edit nutrition data for multiple products at once with the Bulk Editor.', 'woonutri' ); ?></p>
                <ul class="feature-list">
                    <li>✓ <?php _e( 'Edit all products in a spreadsheet-like interface', 'woonutri' ); ?></li>
                    <li>✓ <?php _e( 'Apply templates to multiple products', 'woonutri' ); ?></li>
                    <li>✓ <?php _e( 'Import/Export nutrition data via CSV', 'woonutri' ); ?></li>
                    <li>✓ <?php _e( 'Filter by category, status, or search', 'woonutri' ); ?></li>
                    <li>✓ <?php _e( 'Pre-built nutrition templates for common foods', 'woonutri' ); ?></li>
                </ul>
                <a href="<?php echo esc_url( woonutri_fs()->get_upgrade_url() ); ?>" class="button button-primary button-hero">
                    <?php _e( 'Upgrade to Pro', 'woonutri' ); ?>
                </a>
            </div>
        </div>
        <?php
    }

    public static function ajax_get_products_data() {
        check_ajax_referer( 'woonutri_bulk', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        $category = isset( $_POST['category'] ) ? intval( $_POST['category'] ) : 0;
        $status = isset( $_POST['status'] ) ? sanitize_text_field( $_POST['status'] ) : '';
        $search = isset( $_POST['search'] ) ? sanitize_text_field( $_POST['search'] ) : '';
        $page = isset( $_POST['page'] ) ? intval( $_POST['page'] ) : 1;
        $per_page = 50;

        $args = [
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => $per_page,
            'paged' => $page,
            'orderby' => 'title',
            'order' => 'ASC',
        ];

        if ( $category > 0 ) {
            $args['tax_query'] = [
                [
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $category,
                ],
            ];
        }

        if ( ! empty( $search ) ) {
            $args['s'] = $search;
        }

        if ( $status === 'with' ) {
            $args['meta_query'] = [
                [
                    'key' => 'woonutri_energy_kcal',
                    'compare' => 'EXISTS',
                ],
            ];
        } elseif ( $status === 'without' ) {
            $args['meta_query'] = [
                [
                    'key' => 'woonutri_energy_kcal',
                    'compare' => 'NOT EXISTS',
                ],
            ];
        }

        $query = new WP_Query( $args );
        $products = [];

        if ( $query->have_posts() ) {
            while ( $query->have_posts() ) {
                $query->the_post();
                $product_id = get_the_ID();
                $nutrition_data = Woonutri_Meta_Handler::get_nutrition_data( $product_id );

                $products[] = [
                    'id' => $product_id,
                    'title' => get_the_title(),
                    'edit_url' => get_edit_post_link( $product_id, 'raw' ),
                    'thumbnail' => get_the_post_thumbnail_url( $product_id, 'thumbnail' ),
                    'nutrition' => $nutrition_data,
                ];
            }
            wp_reset_postdata();
        }

        wp_send_json_success( [
            'products' => $products,
            'total' => $query->found_posts,
            'pages' => $query->max_num_pages,
            'current_page' => $page,
        ] );
    }

    public static function ajax_bulk_save() {
        check_ajax_referer( 'woonutri_bulk', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        $products_data = isset( $_POST['products'] ) ? $_POST['products'] : [];

        if ( empty( $products_data ) ) {
            wp_send_json_error( [ 'message' => __( 'No data to save', 'woonutri' ) ] );
        }

        $saved = 0;
        $errors = [];

        foreach ( $products_data as $product ) {
            $product_id = isset( $product['id'] ) ? intval( $product['id'] ) : 0;
            $nutrition = isset( $product['nutrition'] ) ? $product['nutrition'] : [];

            if ( $product_id <= 0 ) {
                continue;
            }

            $sanitized_data = [];
            foreach ( $nutrition as $key => $value ) {
                $sanitized_data[ $key ] = sanitize_text_field( $value );
            }

            try {
                Woonutri_Meta_Handler::set_nutrition_data( $product_id, $sanitized_data );
                $saved++;
            } catch ( Exception $e ) {
                $errors[] = sprintf( __( 'Error saving product %d', 'woonutri' ), $product_id );
            }
        }

        Woonutri_Cache::invalidate_dashboard();

        wp_send_json_success( [
            'saved' => $saved,
            'errors' => $errors,
            'message' => sprintf( __( 'Saved %d products', 'woonutri' ), $saved ),
        ] );
    }

    public static function ajax_bulk_export() {
        check_ajax_referer( 'woonutri_bulk', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        $product_ids = isset( $_POST['product_ids'] ) ? array_map( 'intval', $_POST['product_ids'] ) : [];

        if ( empty( $product_ids ) ) {
            $products = get_posts( [
                'post_type' => 'product',
                'post_status' => 'publish',
                'posts_per_page' => -1,
                'fields' => 'ids',
            ] );
            $product_ids = $products;
        }

        $csv_data = [];
        $headers = [ 'product_id', 'product_name', 'sku' ];
        $fields = Woonutri_Meta_Handler::$fields;
        $headers = array_merge( $headers, $fields );

        $csv_data[] = $headers;

        foreach ( $product_ids as $product_id ) {
            $product = wc_get_product( $product_id );
            if ( ! $product ) {
                continue;
            }

            $row = [
                $product_id,
                $product->get_name(),
                $product->get_sku(),
            ];

            $nutrition = Woonutri_Meta_Handler::get_nutrition_data( $product_id );

            foreach ( $fields as $field ) {
                $row[] = isset( $nutrition[ $field ] ) ? $nutrition[ $field ] : '';
            }

            $csv_data[] = $row;
        }

        wp_send_json_success( [
            'csv' => $csv_data,
            'filename' => 'woonutri-export-' . date( 'Y-m-d' ) . '.csv',
        ] );
    }

    public static function ajax_bulk_import() {
        check_ajax_referer( 'woonutri_bulk', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        $csv_data = isset( $_POST['csv_data'] ) ? $_POST['csv_data'] : [];

        if ( empty( $csv_data ) || count( $csv_data ) < 2 ) {
            wp_send_json_error( [ 'message' => __( 'No valid data in CSV', 'woonutri' ) ] );
        }

        $headers = array_map( 'trim', $csv_data[0] );
        $imported = 0;
        $errors = [];

        $id_col = array_search( 'product_id', $headers );
        $sku_col = array_search( 'sku', $headers );

        if ( $id_col === false && $sku_col === false ) {
            wp_send_json_error( [ 'message' => __( 'CSV must contain product_id or sku column', 'woonutri' ) ] );
        }

        for ( $i = 1; $i < count( $csv_data ); $i++ ) {
            $row = $csv_data[ $i ];
            $product_id = 0;

            if ( $id_col !== false && ! empty( $row[ $id_col ] ) ) {
                $product_id = intval( $row[ $id_col ] );
            } elseif ( $sku_col !== false && ! empty( $row[ $sku_col ] ) ) {
                $product_id = wc_get_product_id_by_sku( trim( $row[ $sku_col ] ) );
            }

            if ( ! $product_id ) {
                $errors[] = sprintf( __( 'Row %d: Product not found', 'woonutri' ), $i + 1 );
                continue;
            }

            $nutrition_data = [];
            foreach ( $headers as $col_index => $header ) {
                if ( in_array( $header, Woonutri_Meta_Handler::$fields ) ) {
                    $value = isset( $row[ $col_index ] ) ? sanitize_text_field( $row[ $col_index ] ) : '';
                    if ( $value !== '' ) {
                        $nutrition_data[ $header ] = $value;
                    }
                }
            }

            if ( ! empty( $nutrition_data ) ) {
                Woonutri_Meta_Handler::set_nutrition_data( $product_id, $nutrition_data );
                $imported++;
            }
        }

        Woonutri_Cache::invalidate_dashboard();

        wp_send_json_success( [
            'imported' => $imported,
            'errors' => $errors,
            'message' => sprintf( __( 'Imported %d products', 'woonutri' ), $imported ),
        ] );
    }

    public static function ajax_apply_template() {
        check_ajax_referer( 'woonutri_bulk', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        $template_key = isset( $_POST['template'] ) ? sanitize_key( $_POST['template'] ) : '';
        $product_ids = isset( $_POST['product_ids'] ) ? array_map( 'intval', $_POST['product_ids'] ) : [];

        if ( empty( $template_key ) || empty( $product_ids ) ) {
            wp_send_json_error( [ 'message' => __( 'Invalid template or no products selected', 'woonutri' ) ] );
        }

        $templates = self::get_templates();
        if ( ! isset( $templates[ $template_key ] ) ) {
            wp_send_json_error( [ 'message' => __( 'Template not found', 'woonutri' ) ] );
        }

        $template_data = $templates[ $template_key ]['data'];
        $applied = 0;

        foreach ( $product_ids as $product_id ) {
            if ( $template_key === 'blank' ) {
                foreach ( Woonutri_Meta_Handler::$fields as $field ) {
                    delete_post_meta( $product_id, 'woonutri_' . $field );
                }
            } else {
                Woonutri_Meta_Handler::set_nutrition_data( $product_id, $template_data );
            }
            $applied++;
        }

        Woonutri_Cache::invalidate_dashboard();

        wp_send_json_success( [
            'applied' => $applied,
            'message' => sprintf( __( 'Applied template to %d products', 'woonutri' ), $applied ),
        ] );
    }
}
