<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Woonutri_Dashboard {

    public static function init() {
        add_action( 'admin_menu', [ __CLASS__, 'add_dashboard_page' ], 5 );
        add_action( 'admin_enqueue_scripts', [ __CLASS__, 'enqueue_dashboard_assets' ] );
        add_action( 'wp_ajax_woonutri_get_dashboard_stats', [ __CLASS__, 'ajax_get_stats' ] );
        add_action( 'wp_ajax_woonutri_dismiss_notice', [ __CLASS__, 'ajax_dismiss_notice' ] );
        add_action( 'wp_ajax_woonutri_save_category_filter', [ __CLASS__, 'ajax_save_category_filter' ] );
    }

    public static function add_dashboard_page() {
        add_menu_page(
            __( 'WooNutri', 'woonutri' ),
            __( 'WooNutri', 'woonutri' ),
            'manage_woocommerce',
            'woonutri-dashboard',
            [ __CLASS__, 'render_dashboard_page' ],
            'dashicons-carrot',
            56
        );

        add_submenu_page(
            'woonutri-dashboard',
            __( 'Dashboard', 'woonutri' ),
            __( 'Dashboard', 'woonutri' ),
            'manage_woocommerce',
            'woonutri-dashboard',
            [ __CLASS__, 'render_dashboard_page' ]
        );
    }

    public static function enqueue_dashboard_assets( $hook ) {
        if ( $hook !== 'toplevel_page_woonutri-dashboard' ) {
            return;
        }

        wp_enqueue_style(
            'woonutri-dashboard',
            WOONUTRI_PLUGIN_URL . 'assets/css/dashboard.css',
            [],
            WOONUTRI_VERSION
        );

        wp_enqueue_script(
            'woonutri-dashboard',
            WOONUTRI_PLUGIN_URL . 'assets/js/dashboard.js',
            [ 'jquery' ],
            WOONUTRI_VERSION,
            true
        );

        wp_localize_script( 'woonutri-dashboard', 'woonutriDashboard', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'woonutri_dashboard' ),
        ] );
    }

    public static function get_category_filter() {
        $options = get_option( 'woonutri_options', [] );
        return [
            'mode' => isset( $options['category_filter_mode'] ) ? $options['category_filter_mode'] : 'all',
            'categories' => isset( $options['category_filter_ids'] ) ? array_map( 'intval', (array) $options['category_filter_ids'] ) : [],
        ];
    }

    public static function get_filtered_product_ids() {
        global $wpdb;

        $filter = self::get_category_filter();

        if ( $filter['mode'] === 'all' || empty( $filter['categories'] ) ) {
            return null;
        }

        $cache_key = 'filtered_ids_' . md5( json_encode( $filter ) );
        $cached_ids = Woonutri_Cache::get( $cache_key );

        if ( false !== $cached_ids ) {
            return $cached_ids;
        }

        $category_ids = implode( ',', array_map( 'intval', $filter['categories'] ) );

        if ( $filter['mode'] === 'include' ) {
            $product_ids = $wpdb->get_col( "
                SELECT DISTINCT p.ID
                FROM {$wpdb->posts} p
                INNER JOIN {$wpdb->term_relationships} tr ON p.ID = tr.object_id
                INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                WHERE p.post_type = 'product'
                AND p.post_status = 'publish'
                AND tt.taxonomy = 'product_cat'
                AND tt.term_id IN ({$category_ids})
            " );
        } else {
            $product_ids = $wpdb->get_col( "
                SELECT DISTINCT p.ID
                FROM {$wpdb->posts} p
                WHERE p.post_type = 'product'
                AND p.post_status = 'publish'
                AND p.ID NOT IN (
                    SELECT DISTINCT tr.object_id
                    FROM {$wpdb->term_relationships} tr
                    INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                    WHERE tt.taxonomy = 'product_cat'
                    AND tt.term_id IN ({$category_ids})
                )
            " );
        }

        Woonutri_Cache::set( $cache_key, $product_ids );

        return $product_ids;
    }

    public static function get_stats() {
        global $wpdb;

        $filter = self::get_category_filter();
        $cache_key = 'dashboard_stats_' . md5( serialize( $filter ) );
        $stats = Woonutri_Cache::get( $cache_key );

        if ( false === $stats ) {
            $filtered_ids = self::get_filtered_product_ids();

            if ( $filtered_ids === null ) {
                $total_products = wp_count_posts( 'product' );
                $total_published = isset( $total_products->publish ) ? $total_products->publish : 0;

                $products_with_nutrition = $wpdb->get_var( "
                    SELECT COUNT(DISTINCT post_id)
                    FROM {$wpdb->postmeta}
                    WHERE meta_key = 'woonutri_energy_kcal'
                    AND meta_value != ''
                    AND meta_value IS NOT NULL
                    AND post_id IN (
                        SELECT ID FROM {$wpdb->posts}
                        WHERE post_type = 'product'
                        AND post_status = 'publish'
                    )
                " );
            } else {
                $total_published = count( $filtered_ids );

                if ( $total_published > 0 ) {
                    $ids_list = implode( ',', array_map( 'intval', $filtered_ids ) );
                    $products_with_nutrition = $wpdb->get_var( "
                        SELECT COUNT(DISTINCT post_id)
                        FROM {$wpdb->postmeta}
                        WHERE meta_key = 'woonutri_energy_kcal'
                        AND meta_value != ''
                        AND meta_value IS NOT NULL
                        AND post_id IN ({$ids_list})
                    " );
                } else {
                    $products_with_nutrition = 0;
                }
            }

            $products_missing = $total_published - $products_with_nutrition;

            $compliance_stats = self::get_compliance_stats( $filtered_ids );

            $nutri_score_distribution = self::get_nutri_score_distribution( $filtered_ids );

            $stats = [
                'total_products' => $total_published,
                'with_nutrition' => (int) $products_with_nutrition,
                'missing_nutrition' => max( 0, $products_missing ),
                'coverage_percentage' => $total_published > 0
                    ? round( ( $products_with_nutrition / $total_published ) * 100, 1 )
                    : 0,
                'compliance' => $compliance_stats,
                'nutri_scores' => $nutri_score_distribution,
                'filter_active' => $filter['mode'] !== 'all' && ! empty( $filter['categories'] ),
                'filter_mode' => $filter['mode'],
            ];

            Woonutri_Cache::set( $cache_key, $stats );
        }

        return $stats;
    }

    private static function get_compliance_stats( $filtered_ids = null ) {
        global $wpdb;

        $options = get_option( 'woonutri_options', [] );
        $region = isset( $options['region'] ) ? $options['region'] : 'eu';

        $required_fields = self::get_required_fields( $region );

        $filter_clause = '';
        if ( $filtered_ids !== null && ! empty( $filtered_ids ) ) {
            $ids_list = implode( ',', array_map( 'intval', $filtered_ids ) );
            $filter_clause = " AND p.ID IN ({$ids_list})";
        }

        $products = $wpdb->get_results( "
            SELECT DISTINCT p.ID
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            AND pm.meta_key = 'woonutri_energy_kcal'
            AND pm.meta_value != ''
            {$filter_clause}
        " );

        if ( empty( $products ) ) {
            return [
                'compliant' => 0,
                'partial' => 0,
                'non_compliant' => 0,
                'total' => 0,
                'compliance_rate' => 0,
                'region' => $region,
                'required_fields' => $required_fields,
            ];
        }

        $product_ids = array_map( function( $p ) { return $p->ID; }, $products );
        $ids_placeholder = implode( ',', array_map( 'intval', $product_ids ) );

        $meta_keys = array_map( function( $field ) { return 'woonutri_' . $field; }, $required_fields );
        $meta_keys_placeholder = "'" . implode( "','", array_map( 'esc_sql', $meta_keys ) ) . "'";

        $all_meta = $wpdb->get_results( "
            SELECT post_id, meta_key, meta_value
            FROM {$wpdb->postmeta}
            WHERE post_id IN ({$ids_placeholder})
            AND meta_key IN ({$meta_keys_placeholder})
        " );

        $meta_by_product = [];
        foreach ( $all_meta as $meta ) {
            if ( ! isset( $meta_by_product[ $meta->post_id ] ) ) {
                $meta_by_product[ $meta->post_id ] = [];
            }
            $meta_by_product[ $meta->post_id ][ $meta->meta_key ] = $meta->meta_value;
        }

        $compliant = 0;
        $partial = 0;
        $non_compliant = 0;

        foreach ( $product_ids as $product_id ) {
            $filled_required = 0;
            $product_meta = isset( $meta_by_product[ $product_id ] ) ? $meta_by_product[ $product_id ] : [];

            foreach ( $required_fields as $field ) {
                $meta_key = 'woonutri_' . $field;
                $value = isset( $product_meta[ $meta_key ] ) ? $product_meta[ $meta_key ] : '';
                if ( ! empty( $value ) && $value !== '0' ) {
                    $filled_required++;
                }
            }

            $compliance_rate = count( $required_fields ) > 0
                ? ( $filled_required / count( $required_fields ) ) * 100
                : 100;

            if ( $compliance_rate >= 100 ) {
                $compliant++;
            } elseif ( $compliance_rate >= 50 ) {
                $partial++;
            } else {
                $non_compliant++;
            }
        }

        $total = count( $products );

        return [
            'compliant' => $compliant,
            'partial' => $partial,
            'non_compliant' => $non_compliant,
            'total' => $total,
            'compliance_rate' => $total > 0 ? round( ( $compliant / $total ) * 100, 1 ) : 0,
            'region' => $region,
            'required_fields' => $required_fields,
        ];
    }

    private static function get_required_fields( $region ) {
        $fields = [
            'eu' => [ 'energy_kcal', 'fat', 'saturated_fat', 'carbs', 'sugar', 'protein', 'salt' ],
            'us' => [ 'energy_kcal', 'fat', 'saturated_fat', 'carbs', 'fiber', 'sugar', 'protein', 'sodium' ],
            'uk' => [ 'energy_kcal', 'fat', 'saturated_fat', 'carbs', 'sugar', 'protein', 'salt' ],
        ];

        return isset( $fields[ $region ] ) ? $fields[ $region ] : $fields['eu'];
    }

    private static function get_nutri_score_distribution( $filtered_ids = null ) {
        global $wpdb;

        $filter_clause = '';
        if ( $filtered_ids !== null && ! empty( $filtered_ids ) ) {
            $ids_list = implode( ',', array_map( 'intval', $filtered_ids ) );
            $filter_clause = " AND post_id IN ({$ids_list})";
        }

        $results = $wpdb->get_results( "
            SELECT meta_value as grade, COUNT(*) as count
            FROM {$wpdb->postmeta}
            WHERE meta_key = 'woonutri_nutri_score'
            AND meta_value IN ('A', 'B', 'C', 'D', 'E')
            {$filter_clause}
            GROUP BY meta_value
            ORDER BY meta_value ASC
        " );

        $distribution = [
            'A' => 0,
            'B' => 0,
            'C' => 0,
            'D' => 0,
            'E' => 0,
        ];

        foreach ( $results as $row ) {
            if ( isset( $distribution[ $row->grade ] ) ) {
                $distribution[ $row->grade ] = (int) $row->count;
            }
        }

        return $distribution;
    }

    public static function get_products_needing_attention( $limit = 5 ) {
        global $wpdb;

        $options = get_option( 'woonutri_options', [] );
        $region = isset( $options['region'] ) ? $options['region'] : 'eu';
        $required_fields = self::get_required_fields( $region );

        $filtered_ids = self::get_filtered_product_ids();

        $filter_clause = '';
        if ( $filtered_ids !== null ) {
            if ( empty( $filtered_ids ) ) {
                return [];
            }
            $ids_list = implode( ',', array_map( 'intval', $filtered_ids ) );
            $filter_clause = " AND p.ID IN ({$ids_list})";
        }

        $products = $wpdb->get_results( $wpdb->prepare( "
            SELECT p.ID, p.post_title
            FROM {$wpdb->posts} p
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            {$filter_clause}
            AND p.ID NOT IN (
                SELECT DISTINCT post_id
                FROM {$wpdb->postmeta}
                WHERE meta_key = 'woonutri_energy_kcal'
                AND meta_value != ''
                AND meta_value IS NOT NULL
            )
            ORDER BY p.post_date DESC
            LIMIT %d
        ", $limit ) );

        $result = [];
        foreach ( $products as $product ) {
            $result[] = [
                'id' => $product->ID,
                'title' => $product->post_title,
                'edit_url' => get_edit_post_link( $product->ID, 'raw' ),
                'missing_fields' => $required_fields,
            ];
        }

        return $result;
    }

    public static function render_dashboard_page() {
        $stats = self::get_stats();
        $products_needing_attention = self::get_products_needing_attention();
        $is_pro = function_exists( 'woonutri_fs' ) && woonutri_fs()->can_use_premium_code();
        $wizard_completed = get_option( 'woonutri_wizard_completed' );
        $category_filter = self::get_category_filter();
        $product_categories = get_terms( [
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC',
        ] );
        ?>
        <div class="wrap woonutri-dashboard-wrap">
            <div class="woonutri-dashboard-header">
                <div class="woonutri-dashboard-title">
                    <span class="woonutri-logo">🥗</span>
                    <h1><?php _e( 'WooNutri Dashboard', 'woonutri' ); ?></h1>
                    <?php if ( $is_pro ) : ?>
                    <span class="woonutri-pro-badge">PRO</span>
                    <?php endif; ?>
                </div>
                <div class="woonutri-dashboard-actions">
                    <?php if ( ! $wizard_completed ) : ?>
                    <a href="<?php echo admin_url( 'admin.php?page=woonutri-setup' ); ?>" class="button">
                        <?php _e( 'Run Setup Wizard', 'woonutri' ); ?>
                    </a>
                    <?php endif; ?>
                    <a href="<?php echo admin_url( 'admin.php?page=woonutri-settings' ); ?>" class="button">
                        <span class="dashicons dashicons-admin-settings"></span>
                        <?php _e( 'Settings', 'woonutri' ); ?>
                    </a>
                </div>
            </div>

            <div class="woonutri-category-filter-card">
                <div class="filter-header">
                    <span class="filter-icon">🏷️</span>
                    <h3><?php _e( 'Product Category Filter', 'woonutri' ); ?></h3>
                    <p class="filter-description"><?php _e( 'Filter dashboard statistics to only show products that need nutrition data (e.g., exclude clothing, accessories).', 'woonutri' ); ?></p>
                </div>
                <div class="filter-content">
                    <div class="filter-mode">
                        <label>
                            <input type="radio" name="category_filter_mode" value="all" <?php checked( $category_filter['mode'], 'all' ); ?> />
                            <?php _e( 'All Products', 'woonutri' ); ?>
                            <span class="mode-description"><?php _e( 'Include all products in statistics', 'woonutri' ); ?></span>
                        </label>
                        <label>
                            <input type="radio" name="category_filter_mode" value="include" <?php checked( $category_filter['mode'], 'include' ); ?> />
                            <?php _e( 'Include Only', 'woonutri' ); ?>
                            <span class="mode-description"><?php _e( 'Only include products from selected categories', 'woonutri' ); ?></span>
                        </label>
                        <label>
                            <input type="radio" name="category_filter_mode" value="exclude" <?php checked( $category_filter['mode'], 'exclude' ); ?> />
                            <?php _e( 'Exclude Categories', 'woonutri' ); ?>
                            <span class="mode-description"><?php _e( 'Exclude products from selected categories', 'woonutri' ); ?></span>
                        </label>
                    </div>
                    <div class="filter-categories" style="<?php echo $category_filter['mode'] === 'all' ? 'display:none;' : ''; ?>">
                        <label class="categories-label"><?php _e( 'Select Categories:', 'woonutri' ); ?></label>
                        <div class="categories-grid">
                            <?php if ( ! is_wp_error( $product_categories ) ) : ?>
                                <?php foreach ( $product_categories as $cat ) : ?>
                                <label class="category-checkbox">
                                    <input type="checkbox" name="category_filter_ids[]" value="<?php echo esc_attr( $cat->term_id ); ?>"
                                        <?php checked( in_array( $cat->term_id, $category_filter['categories'] ) ); ?> />
                                    <span class="category-name"><?php echo esc_html( $cat->name ); ?></span>
                                    <span class="category-count">(<?php echo $cat->count; ?>)</span>
                                </label>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="filter-actions">
                        <button type="button" class="button button-primary" id="woonutri-save-filter">
                            <?php _e( 'Apply Filter', 'woonutri' ); ?>
                        </button>
                        <?php if ( $category_filter['mode'] !== 'all' && ! empty( $category_filter['categories'] ) ) : ?>
                        <span class="filter-status active">
                            <span class="dashicons dashicons-filter"></span>
                            <?php
                            printf(
                                __( 'Filter active: %s %d categories', 'woonutri' ),
                                $category_filter['mode'] === 'include' ? __( 'Including', 'woonutri' ) : __( 'Excluding', 'woonutri' ),
                                count( $category_filter['categories'] )
                            );
                            ?>
                        </span>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <div class="woonutri-stats-grid">
                <div class="woonutri-stat-card">
                    <div class="stat-icon" style="background: #10b981;">📦</div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format( $stats['total_products'] ); ?></div>
                        <div class="stat-label"><?php _e( 'Total Products', 'woonutri' ); ?></div>
                    </div>
                </div>

                <div class="woonutri-stat-card">
                    <div class="stat-icon" style="background: #3b82f6;">✓</div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format( $stats['with_nutrition'] ); ?></div>
                        <div class="stat-label"><?php _e( 'With Nutrition Data', 'woonutri' ); ?></div>
                    </div>
                </div>

                <div class="woonutri-stat-card <?php echo $stats['missing_nutrition'] > 0 ? 'warning' : ''; ?>">
                    <div class="stat-icon" style="background: <?php echo $stats['missing_nutrition'] > 0 ? '#f59e0b' : '#10b981'; ?>;">
                        <?php echo $stats['missing_nutrition'] > 0 ? '⚠️' : '✓'; ?>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format( $stats['missing_nutrition'] ); ?></div>
                        <div class="stat-label"><?php _e( 'Missing Data', 'woonutri' ); ?></div>
                    </div>
                </div>

                <div class="woonutri-stat-card">
                    <div class="stat-icon" style="background: #8b5cf6;">📊</div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $stats['coverage_percentage']; ?>%</div>
                        <div class="stat-label"><?php _e( 'Coverage', 'woonutri' ); ?></div>
                    </div>
                    <div class="stat-progress">
                        <div class="stat-progress-bar" style="width: <?php echo $stats['coverage_percentage']; ?>%;"></div>
                    </div>
                </div>
            </div>

            <div class="woonutri-dashboard-grid">
                <div class="woonutri-dashboard-card">
                    <div class="card-header">
                        <h2><?php _e( 'Compliance Status', 'woonutri' ); ?></h2>
                        <span class="region-badge"><?php echo strtoupper( $stats['compliance']['region'] ); ?></span>
                    </div>
                    <div class="card-content">
                        <div class="compliance-meter">
                            <div class="compliance-circle" data-percentage="<?php echo $stats['compliance']['compliance_rate']; ?>">
                                <svg viewBox="0 0 36 36">
                                    <path class="circle-bg"
                                        d="M18 2.0845
                                        a 15.9155 15.9155 0 0 1 0 31.831
                                        a 15.9155 15.9155 0 0 1 0 -31.831"
                                        fill="none" stroke="#e5e7eb" stroke-width="3"/>
                                    <path class="circle-progress"
                                        d="M18 2.0845
                                        a 15.9155 15.9155 0 0 1 0 31.831
                                        a 15.9155 15.9155 0 0 1 0 -31.831"
                                        fill="none"
                                        stroke="<?php echo $stats['compliance']['compliance_rate'] >= 80 ? '#10b981' : ( $stats['compliance']['compliance_rate'] >= 50 ? '#f59e0b' : '#ef4444' ); ?>"
                                        stroke-width="3"
                                        stroke-dasharray="<?php echo $stats['compliance']['compliance_rate']; ?>, 100"/>
                                </svg>
                                <div class="circle-text">
                                    <span class="percentage"><?php echo $stats['compliance']['compliance_rate']; ?>%</span>
                                    <span class="label"><?php _e( 'Compliant', 'woonutri' ); ?></span>
                                </div>
                            </div>
                        </div>
                        <div class="compliance-breakdown">
                            <div class="breakdown-item compliant">
                                <span class="dot"></span>
                                <span class="label"><?php _e( 'Fully Compliant', 'woonutri' ); ?></span>
                                <span class="value"><?php echo $stats['compliance']['compliant']; ?></span>
                            </div>
                            <div class="breakdown-item partial">
                                <span class="dot"></span>
                                <span class="label"><?php _e( 'Partial', 'woonutri' ); ?></span>
                                <span class="value"><?php echo $stats['compliance']['partial']; ?></span>
                            </div>
                            <div class="breakdown-item non-compliant">
                                <span class="dot"></span>
                                <span class="label"><?php _e( 'Non-Compliant', 'woonutri' ); ?></span>
                                <span class="value"><?php echo $stats['compliance']['non_compliant']; ?></span>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="woonutri-dashboard-card">
                    <div class="card-header">
                        <h2><?php _e( 'Nutri-Score Distribution', 'woonutri' ); ?></h2>
                    </div>
                    <div class="card-content">
                        <div class="nutri-score-chart">
                            <?php
                            $colors = [
                                'A' => '#038141',
                                'B' => '#85BB2F',
                                'C' => '#FECB02',
                                'D' => '#EE8100',
                                'E' => '#E63946',
                            ];
                            $total_scores = array_sum( $stats['nutri_scores'] );
                            foreach ( $stats['nutri_scores'] as $grade => $count ) :
                                $percentage = $total_scores > 0 ? ( $count / $total_scores ) * 100 : 0;
                            ?>
                            <div class="nutri-score-bar">
                                <div class="bar-label">
                                    <span class="grade" style="background: <?php echo $colors[ $grade ]; ?>;"><?php echo $grade; ?></span>
                                </div>
                                <div class="bar-track">
                                    <div class="bar-fill" style="width: <?php echo $percentage; ?>%; background: <?php echo $colors[ $grade ]; ?>;"></div>
                                </div>
                                <div class="bar-value"><?php echo $count; ?></div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php if ( $total_scores === 0 ) : ?>
                        <p class="no-data-message"><?php _e( 'No products have Nutri-Score calculated yet.', 'woonutri' ); ?></p>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="woonutri-dashboard-card full-width">
                    <div class="card-header">
                        <h2><?php _e( 'Products Needing Attention', 'woonutri' ); ?></h2>
                        <a href="<?php echo admin_url( 'edit.php?post_type=product' ); ?>" class="view-all-link">
                            <?php _e( 'View All Products', 'woonutri' ); ?> →
                        </a>
                    </div>
                    <div class="card-content">
                        <?php if ( ! empty( $products_needing_attention ) ) : ?>
                        <table class="woonutri-products-table">
                            <thead>
                                <tr>
                                    <th><?php _e( 'Product', 'woonutri' ); ?></th>
                                    <th><?php _e( 'Missing Fields', 'woonutri' ); ?></th>
                                    <th><?php _e( 'Action', 'woonutri' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $products_needing_attention as $product ) : ?>
                                <tr>
                                    <td>
                                        <strong><?php echo esc_html( $product['title'] ); ?></strong>
                                    </td>
                                    <td>
                                        <span class="missing-fields">
                                            <?php echo implode( ', ', array_map( 'ucwords', str_replace( '_', ' ', $product['missing_fields'] ) ) ); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="<?php echo esc_url( $product['edit_url'] ); ?>" class="button button-small">
                                            <?php _e( 'Edit', 'woonutri' ); ?>
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        <?php else : ?>
                        <div class="no-products-message">
                            <span class="icon">🎉</span>
                            <p><?php _e( 'All products have nutrition data! Great job!', 'woonutri' ); ?></p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <?php if ( $is_pro ) :
                    $analytics_data = Woonutri_Analytics::get_dashboard_summary();
                ?>
                <div class="woonutri-dashboard-card">
                    <div class="card-header">
                        <h2><?php _e( 'Label Analytics', 'woonutri' ); ?></h2>
                        <span class="woonutri-pro-badge">PRO</span>
                    </div>
                    <div class="card-content">
                        <div class="analytics-stats-row">
                            <div class="analytics-stat">
                                <div class="stat-value"><?php echo number_format( $analytics_data['total_views_7d'] ); ?></div>
                                <div class="stat-label"><?php _e( 'Views (7 days)', 'woonutri' ); ?></div>
                            </div>
                            <div class="analytics-stat">
                                <div class="stat-value"><?php echo number_format( $analytics_data['total_views_30d'] ); ?></div>
                                <div class="stat-label"><?php _e( 'Views (30 days)', 'woonutri' ); ?></div>
                            </div>
                            <div class="analytics-stat">
                                <div class="stat-value"><?php echo number_format( $analytics_data['total_views_all'] ); ?></div>
                                <div class="stat-label"><?php _e( 'All Time', 'woonutri' ); ?></div>
                            </div>
                        </div>
                        <?php if ( ! empty( $analytics_data['top_products'] ) ) : ?>
                        <div class="analytics-top-products">
                            <h4><?php _e( 'Most Viewed Labels', 'woonutri' ); ?></h4>
                            <ul>
                                <?php foreach ( array_slice( $analytics_data['top_products'], 0, 3 ) as $product ) : ?>
                                <li>
                                    <span class="product-name"><?php echo esc_html( $product['name'] ); ?></span>
                                    <span class="product-views"><?php echo number_format( $product['views'] ); ?> <?php _e( 'views', 'woonutri' ); ?></span>
                                </li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                        <?php else : ?>
                        <p class="analytics-no-data"><?php _e( 'No label views recorded yet. Views will appear as customers view your product nutrition labels.', 'woonutri' ); ?></p>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>

                <div class="woonutri-dashboard-card">
                    <div class="card-header">
                        <h2><?php _e( 'Quick Actions', 'woonutri' ); ?></h2>
                    </div>
                    <div class="card-content">
                        <div class="quick-actions-grid">
                            <a href="<?php echo admin_url( 'post-new.php?post_type=product' ); ?>" class="quick-action-item">
                                <span class="action-icon">➕</span>
                                <span class="action-label"><?php _e( 'Add Product', 'woonutri' ); ?></span>
                            </a>
                            <a href="<?php echo admin_url( 'admin.php?page=woonutri-settings' ); ?>" class="quick-action-item">
                                <span class="action-icon">⚙️</span>
                                <span class="action-label"><?php _e( 'Settings', 'woonutri' ); ?></span>
                            </a>
                            <?php if ( $is_pro ) : ?>
                            <a href="#" class="quick-action-item" id="woonutri-bulk-import">
                                <span class="action-icon">📥</span>
                                <span class="action-label"><?php _e( 'Import CSV', 'woonutri' ); ?></span>
                            </a>
                            <a href="#" class="quick-action-item" id="woonutri-bulk-export">
                                <span class="action-icon">📤</span>
                                <span class="action-label"><?php _e( 'Export CSV', 'woonutri' ); ?></span>
                            </a>
                            <?php else : ?>
                            <a href="<?php echo esc_url( woonutri_fs()->get_upgrade_url() ); ?>" class="quick-action-item pro-feature">
                                <span class="action-icon">📥</span>
                                <span class="action-label"><?php _e( 'Import CSV', 'woonutri' ); ?></span>
                                <span class="pro-badge">PRO</span>
                            </a>
                            <a href="<?php echo esc_url( woonutri_fs()->get_upgrade_url() ); ?>" class="quick-action-item pro-feature">
                                <span class="action-icon">📤</span>
                                <span class="action-label"><?php _e( 'Export CSV', 'woonutri' ); ?></span>
                                <span class="pro-badge">PRO</span>
                            </a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <?php if ( ! $is_pro ) : ?>
                <div class="woonutri-dashboard-card pro-upsell">
                    <div class="card-content">
                        <div class="upsell-content">
                            <div class="upsell-icon">⭐</div>
                            <div class="upsell-text">
                                <h3><?php _e( 'Upgrade to WooNutri Pro', 'woonutri' ); ?></h3>
                                <p><?php _e( 'Unlock FDA themes, AI-powered OCR scanning, CSV import/export, and more!', 'woonutri' ); ?></p>
                                <ul class="pro-features-list">
                                    <li>✓ <?php _e( '10+ Premium Themes', 'woonutri' ); ?></li>
                                    <li>✓ <?php _e( 'AI Label Scanning', 'woonutri' ); ?></li>
                                    <li>✓ <?php _e( 'Barcode Lookup', 'woonutri' ); ?></li>
                                    <li>✓ <?php _e( 'CSV Import/Export', 'woonutri' ); ?></li>
                                    <li>✓ <?php _e( 'Priority Support', 'woonutri' ); ?></li>
                                </ul>
                            </div>
                            <div class="upsell-cta">
                                <a href="<?php echo esc_url( woonutri_fs()->get_upgrade_url() ); ?>" class="button button-primary button-hero">
                                    <?php _e( 'Upgrade Now', 'woonutri' ); ?>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    public static function ajax_get_stats() {
        check_ajax_referer( 'woonutri_dashboard', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        delete_transient( 'woonutri_dashboard_stats' );
        $stats = self::get_stats();

        wp_send_json_success( $stats );
    }

    public static function ajax_dismiss_notice() {
        check_ajax_referer( 'woonutri_dashboard', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        $notice_id = isset( $_POST['notice_id'] ) ? sanitize_key( $_POST['notice_id'] ) : '';
        if ( $notice_id ) {
            $dismissed = get_option( 'woonutri_dismissed_notices', [] );
            $dismissed[ $notice_id ] = current_time( 'timestamp' );
            update_option( 'woonutri_dismissed_notices', $dismissed );
        }

        wp_send_json_success();
    }

    public static function ajax_save_category_filter() {
        check_ajax_referer( 'woonutri_dashboard', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        $mode = isset( $_POST['mode'] ) ? sanitize_text_field( $_POST['mode'] ) : 'all';
        $categories = isset( $_POST['categories'] ) ? array_map( 'intval', (array) $_POST['categories'] ) : [];

        if ( ! in_array( $mode, [ 'all', 'include', 'exclude' ], true ) ) {
            $mode = 'all';
        }

        $options = get_option( 'woonutri_options', [] );
        $options['category_filter_mode'] = $mode;
        $options['category_filter_ids'] = $categories;
        update_option( 'woonutri_options', $options );

        Woonutri_Cache::invalidate_dashboard();

        wp_send_json_success( [
            'message' => __( 'Filter saved successfully', 'woonutri' ),
            'mode' => $mode,
            'categories' => $categories,
        ] );
    }
}
