<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Woonutri_Settings
{

    
    private static $options_cache = null;

    
    public static function get_options()
    {
        if (null === self::$options_cache) {
            self::$options_cache = get_option('woonutri_options', []);
        }
        return self::$options_cache;
    }

    
    public static function clear_options_cache()
    {
        self::$options_cache = null;
    }

    public static function init()
    {
        add_action('admin_menu', [__CLASS__, 'add_settings_page']);
        add_action('admin_init', [__CLASS__, 'register_settings']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin_scripts']);
        add_action('wp_ajax_woonutri_get_preview', [__CLASS__, 'ajax_get_preview']);
        add_action('wp_ajax_woonutri_dismiss_notice', [__CLASS__, 'ajax_dismiss_notice']);
        add_action('wp_ajax_woonutri_clear_cache', [__CLASS__, 'ajax_clear_cache']);
    }

    public static function add_settings_page()
    {
        
        add_submenu_page(
            'woonutri-dashboard',
            __('Settings', 'smart-nutrition-labels'),
            __('Settings', 'smart-nutrition-labels'),
            'manage_woocommerce',
            'woonutri-settings',
            [__CLASS__, 'render_settings_page']
        );
    }

    public static function register_settings()
    {
        register_setting('woonutri_options', 'woonutri_options', [__CLASS__, 'sanitize_options']);

        
        
        global $pagenow;
        $is_settings_page = ($pagenow === 'admin.php' && isset($_GET['page']) && $_GET['page'] === 'woonutri-settings');
        
        if (!$is_settings_page && !wp_doing_ajax()) {
            return;
        }

        
        add_settings_section(
            'woonutri_display',
            __('Display Position', 'smart-nutrition-labels'),
            '__return_false',
            'woonutri_options'
        );
        add_settings_field(
            'display_position',
            __('Display Location', 'smart-nutrition-labels'),
            [__CLASS__, 'display_position_field'],
            'woonutri_options',
            'woonutri_display'
        );
        add_settings_field(
            'tab_title',
            __('Tab Title', 'smart-nutrition-labels'),
            [__CLASS__, 'tab_title_field'],
            'woonutri_options',
            'woonutri_display'
        );
        add_settings_field(
            'tab_priority',
            __('Tab Priority', 'smart-nutrition-labels'),
            [__CLASS__, 'tab_priority_field'],
            'woonutri_options',
            'woonutri_display'
        );
        add_settings_field(
            'tab_show_icon',
            __('Show Tab Icon', 'smart-nutrition-labels'),
            [__CLASS__, 'tab_show_icon_field'],
            'woonutri_options',
            'woonutri_display'
        );

        
        add_settings_section(
            'woonutri_typography',
            __('Typography', 'smart-nutrition-labels'),
            '__return_false',
            'woonutri_options'
        );
        add_settings_field(
            'font_family',
            __('Font Family', 'smart-nutrition-labels'),
            [__CLASS__, 'font_family_field'],
            'woonutri_options',
            'woonutri_typography'
        );
        add_settings_field(
            'custom_font',
            __('Custom Google Font', 'smart-nutrition-labels'),
            [__CLASS__, 'custom_font_field'],
            'woonutri_options',
            'woonutri_typography'
        );
        add_settings_field(
            'font_size',
            __('Font Size', 'smart-nutrition-labels') . ' <span title="' . esc_attr__('Controls the base font size for the label.', 'smart-nutrition-labels') . '">?</span>',
            [__CLASS__, 'font_size_field'],
            'woonutri_options',
            'woonutri_typography'
        );
        
        add_settings_section(
            'woonutri_colors',
            __('Colors', 'smart-nutrition-labels'),
            '__return_false',
            'woonutri_options'
        );
        add_settings_field(
            'primary_color',
            __('Primary Color', 'smart-nutrition-labels'),
            [__CLASS__, 'primary_color_field'],
            'woonutri_options',
            'woonutri_colors'
        );
        add_settings_field(
            'background_color',
            __('Background Color', 'smart-nutrition-labels'),
            [__CLASS__, 'background_color_field'],
            'woonutri_options',
            'woonutri_colors'
        );
        add_settings_field(
            'text_color',
            __('Text Color', 'smart-nutrition-labels'),
            [__CLASS__, 'text_color_field'],
            'woonutri_options',
            'woonutri_colors'
        );
        add_settings_field(
            'header_color',
            __('Header Color', 'smart-nutrition-labels'),
            [__CLASS__, 'header_color_field'],
            'woonutri_options',
            'woonutri_colors'
        );
        add_settings_field(
            'border_color',
            __('Border Color', 'smart-nutrition-labels'),
            [__CLASS__, 'border_color_field'],
            'woonutri_options',
            'woonutri_colors'
        );
        
        add_settings_section(
            'woonutri_layout',
            __('Layout', 'smart-nutrition-labels'),
            '__return_false',
            'woonutri_options'
        );
        add_settings_field(
            'label_theme',
            __('Label Theme', 'smart-nutrition-labels'),
            [__CLASS__, 'label_theme_field'],
            'woonutri_options',
            'woonutri_layout'
        );
        add_settings_field(
            'table_layout',
            __('Table Layout', 'smart-nutrition-labels') . ' <span title="' . esc_attr__('Controls the table padding and density.', 'smart-nutrition-labels') . '">?</span>',
            [__CLASS__, 'table_layout_field'],
            'woonutri_options',
            'woonutri_layout'
        );
        add_settings_field(
            'border_radius',
            __('Border Radius', 'smart-nutrition-labels'),
            [__CLASS__, 'border_radius_field'],
            'woonutri_options',
            'woonutri_layout'
        );
        add_settings_field(
            'show_logo',
            __('Show Logo on Label', 'smart-nutrition-labels'),
            [__CLASS__, 'show_logo_field'],
            'woonutri_options',
            'woonutri_layout'
        );
        add_settings_field(
            'enable_collapsible',
            __('Enable Collapsible Sections', 'smart-nutrition-labels'),
            [__CLASS__, 'enable_collapsible_field'],
            'woonutri_options',
            'woonutri_layout'
        );

        
        add_settings_section(
            'woonutri_pro_styling',
            __('Pro Styling', 'smart-nutrition-labels'),
            [__CLASS__, 'pro_styling_section_callback'],
            'woonutri_options'
        );
        add_settings_field(
            'enable_shadow',
            __('Box Shadow', 'smart-nutrition-labels'),
            [__CLASS__, 'enable_shadow_field'],
            'woonutri_options',
            'woonutri_pro_styling'
        );
        add_settings_field(
            'shadow_intensity',
            __('Shadow Intensity', 'smart-nutrition-labels'),
            [__CLASS__, 'shadow_intensity_field'],
            'woonutri_options',
            'woonutri_pro_styling'
        );
        add_settings_field(
            'enable_gradient',
            __('Gradient Background', 'smart-nutrition-labels'),
            [__CLASS__, 'enable_gradient_field'],
            'woonutri_options',
            'woonutri_pro_styling'
        );
        add_settings_field(
            'gradient_colors',
            __('Gradient Colors', 'smart-nutrition-labels'),
            [__CLASS__, 'gradient_colors_field'],
            'woonutri_options',
            'woonutri_pro_styling'
        );
        add_settings_field(
            'enable_animations',
            __('Hover Animations', 'smart-nutrition-labels'),
            [__CLASS__, 'enable_animations_field'],
            'woonutri_options',
            'woonutri_pro_styling'
        );
        add_settings_field(
            'custom_css',
            __('Custom CSS', 'smart-nutrition-labels'),
            [__CLASS__, 'custom_css_field'],
            'woonutri_options',
            'woonutri_pro_styling'
        );

        
        add_settings_section(
            'woonutri_features',
            __('Features', 'smart-nutrition-labels'),
            [__CLASS__, 'features_section_callback'],
            'woonutri_options'
        );
        add_settings_field(
            'show_pdf_button',
            __('PDF Export Button', 'smart-nutrition-labels'),
            [__CLASS__, 'show_pdf_button_field'],
            'woonutri_options',
            'woonutri_features'
        );
        add_settings_field(
            'show_nutri_score',
            __('Nutri-Score Badge', 'smart-nutrition-labels'),
            [__CLASS__, 'show_nutri_score_field'],
            'woonutri_options',
            'woonutri_features'
        );
        add_settings_field(
            'enable_schema_seo',
            __('Schema.org SEO', 'smart-nutrition-labels'),
            [__CLASS__, 'enable_schema_seo_field'],
            'woonutri_options',
            'woonutri_features'
        );
        add_settings_field(
            'display_format',
            __('Display Format', 'smart-nutrition-labels'),
            [__CLASS__, 'display_format_field'],
            'woonutri_options',
            'woonutri_features'
        );
        add_settings_field(
            'show_traffic_lights',
            __('UK Traffic Light Indicators', 'smart-nutrition-labels'),
            [__CLASS__, 'show_traffic_lights_field'],
            'woonutri_options',
            'woonutri_features'
        );

        
        add_settings_section(
            'woonutri_units',
            __('Custom Units', 'smart-nutrition-labels'),
            '__return_false',
            'woonutri_options'
        );
        foreach ([
            'energy_kcal' => __('Energy', 'smart-nutrition-labels'),
            'fat' => __('Fat', 'smart-nutrition-labels'),
            'saturated_fat' => __('Saturated Fat', 'smart-nutrition-labels'),
            'trans_fat' => __('Trans Fat', 'smart-nutrition-labels'),
            'carbs' => __('Carbohydrates', 'smart-nutrition-labels'),
            'sugar' => __('Sugar', 'smart-nutrition-labels'),
            'added_sugars' => __('Added Sugars', 'smart-nutrition-labels'),
            'protein' => __('Protein', 'smart-nutrition-labels'),
            'fiber' => __('Fiber', 'smart-nutrition-labels'),
            'salt' => __('Salt', 'smart-nutrition-labels'),
            'sodium' => __('Sodium', 'smart-nutrition-labels'),
            'cholesterol' => __('Cholesterol', 'smart-nutrition-labels'),
            'potassium' => __('Potassium', 'smart-nutrition-labels'),
            'vitamin_d' => __('Vitamin D', 'smart-nutrition-labels'),
            'calcium' => __('Calcium', 'smart-nutrition-labels'),
            'iron' => __('Iron', 'smart-nutrition-labels'),
        ] as $key => $label) {
            add_settings_field(
                'unit_' . $key,
                $label . ' ' . __('Unit', 'smart-nutrition-labels'),
                function () use ($key) {
                    self::unit_field($key);
                },
                'woonutri_options',
                'woonutri_units'
            );
        }

        
        add_settings_section(
            'woonutri_labels',
            __('Custom Label Text', 'smart-nutrition-labels'),
            [__CLASS__, 'labels_section_callback'],
            'woonutri_options'
        );
        add_settings_field(
            'label_title',
            __('Label Title', 'smart-nutrition-labels'),
            [__CLASS__, 'label_title_field'],
            'woonutri_options',
            'woonutri_labels'
        );
        add_settings_field(
            'label_per_serving',
            __('Per Serving Text', 'smart-nutrition-labels'),
            [__CLASS__, 'label_per_serving_field'],
            'woonutri_options',
            'woonutri_labels'
        );
        add_settings_field(
            'label_per_100g',
            __('Per 100g Text', 'smart-nutrition-labels'),
            [__CLASS__, 'label_per_100g_field'],
            'woonutri_options',
            'woonutri_labels'
        );
        add_settings_field(
            'label_daily_value',
            __('Daily Value Text', 'smart-nutrition-labels'),
            [__CLASS__, 'label_daily_value_field'],
            'woonutri_options',
            'woonutri_labels'
        );
    }

    public static function sanitize_options($options)
    {
        
        $allowed_positions = ['summary', 'tab', 'both'];
        $options['display_position'] = isset($options['display_position']) && in_array($options['display_position'], $allowed_positions)
            ? $options['display_position'] : 'summary';

        $options['tab_title'] = isset($options['tab_title']) ? substr(sanitize_text_field($options['tab_title']), 0, 50) : 'Nutrition Facts';

        $options['tab_priority'] = isset($options['tab_priority']) ? max(1, min(100, intval($options['tab_priority']))) : 50;

        $options['tab_show_icon'] = isset($options['tab_show_icon']) ? (bool) $options['tab_show_icon'] : true;

        $options['font_family'] = isset($options['font_family']) ? sanitize_text_field($options['font_family']) : 'default';
        $options['custom_font'] = isset($options['custom_font']) ? sanitize_text_field($options['custom_font']) : '';
        $options['font_size'] = isset($options['font_size']) ? sanitize_text_field($options['font_size']) : 'normal';
        $options['primary_color'] = isset($options['primary_color']) ? sanitize_hex_color($options['primary_color']) : '#2a7ae2';
        $options['background_color'] = isset($options['background_color']) ? sanitize_hex_color($options['background_color']) : '#fff';
        $options['text_color'] = isset($options['text_color']) ? sanitize_hex_color($options['text_color']) : '#222';
        $options['header_color'] = isset($options['header_color']) ? sanitize_hex_color($options['header_color']) : '#2a7ae2';
        $options['border_color'] = isset($options['border_color']) ? sanitize_hex_color($options['border_color']) : '#222';
        $options['label_theme'] = isset($options['label_theme']) ? sanitize_text_field($options['label_theme']) : 'basic';

        
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();

        
        $allowed_themes = [
            
            'basic', 'minimal', 'clean', 'simple', 'compact',
            
            'fda', 'eu', 'uk', 'australia', 'canada', 'nutriscore', 'latin', 'gda',
            
            'modern', 'classic', 'elegant', 'bold', 'soft', 'sharp', 'rounded', 'flat',
            
            'organic', 'fitness', 'kids', 'premium', 'eco', 'medical', 'scientific', 'vegan',
            
            'gradient', 'neon', 'retro', 'dark', 'paper', 'glass', 'chalk', 'blueprint',
            
            'us-fda',
        ];

        
        $free_themes = ['basic', 'minimal', 'clean', 'simple', 'compact'];

        if (!in_array($options['label_theme'], $allowed_themes)) {
            $options['label_theme'] = 'basic';
        }

        
        if (!$is_pro && !in_array($options['label_theme'], $free_themes)) {
            $options['label_theme'] = 'basic';
        }
        $options['table_layout'] = isset($options['table_layout']) ? sanitize_text_field($options['table_layout']) : 'standard';
        $options['border_radius'] = isset($options['border_radius']) ? sanitize_text_field($options['border_radius']) : 'medium';
        $options['show_logo'] = isset($options['show_logo']) ? (bool) $options['show_logo'] : false;
        $options['enable_collapsible'] = isset($options['enable_collapsible']) ? (bool) $options['enable_collapsible'] : true;

        
        $options['show_pdf_button'] = isset($options['show_pdf_button']) ? (bool) $options['show_pdf_button'] : false;
        $options['show_nutri_score'] = isset($options['show_nutri_score']) ? (bool) $options['show_nutri_score'] : true;
        $options['enable_schema_seo'] = isset($options['enable_schema_seo']) ? (bool) $options['enable_schema_seo'] : true;
        $options['show_traffic_lights'] = isset($options['show_traffic_lights']) ? (bool) $options['show_traffic_lights'] : false;
        
        $allowed_formats = ['per_100g', 'per_serving', 'both'];
        $options['display_format'] = isset($options['display_format']) && in_array($options['display_format'], $allowed_formats)
            ? $options['display_format'] : 'per_100g';

        
        $unit_defaults = [
            'energy_kcal' => 'kcal',
            'fat' => 'g',
            'saturated_fat' => 'g',
            'trans_fat' => 'g',
            'carbs' => 'g',
            'sugar' => 'g',
            'added_sugars' => 'g',
            'protein' => 'g',
            'fiber' => 'g',
            'salt' => 'g',
            'sodium' => 'mg',
            'cholesterol' => 'mg',
            'potassium' => 'mg',
            'vitamin_d' => 'mcg',
            'calcium' => 'mg',
            'iron' => 'mg',
        ];
        foreach ($unit_defaults as $key => $default) {
            $options['unit_' . $key] = isset($options['unit_' . $key]) ? sanitize_text_field($options['unit_' . $key]) : $default;
        }

        
        $options['label_title'] = isset($options['label_title']) ? sanitize_text_field($options['label_title']) : '';
        $options['label_per_serving'] = isset($options['label_per_serving']) ? sanitize_text_field($options['label_per_serving']) : '';
        $options['label_per_100g'] = isset($options['label_per_100g']) ? sanitize_text_field($options['label_per_100g']) : '';
        $options['label_daily_value'] = isset($options['label_daily_value']) ? sanitize_text_field($options['label_daily_value']) : '';

        
        self::clear_options_cache();

        return $options;
    }

    public static function display_position_field()
    {
        $options = self::get_options();
        $position = isset($options['display_position']) ? $options['display_position'] : 'summary';
        ?>
        <div class="woonutri-radio-group" id="woonutri-display-position">
            <label class="woonutri-radio-card <?php echo $position === 'summary' ? 'checked' : ''; ?>">
                <input type="radio" name="woonutri_options[display_position]" value="summary" <?php checked($position, 'summary'); ?> />
                <div class="woonutri-radio-header">
                    <span class="woonutri-radio-check"></span>
                    <?php _e('Product Summary', 'smart-nutrition-labels'); ?>
                </div>
                <div class="woonutri-radio-desc">
                    <?php _e('Display near the add to cart button.', 'smart-nutrition-labels'); ?>
                </div>
            </label>

            <label class="woonutri-radio-card <?php echo $position === 'tab' ? 'checked' : ''; ?>">
                <input type="radio" name="woonutri_options[display_position]" value="tab" <?php checked($position, 'tab'); ?> />
                <div class="woonutri-radio-header">
                    <span class="woonutri-radio-check"></span>
                    <?php _e('Dedicated Tab', 'smart-nutrition-labels'); ?>
                </div>
                <div class="woonutri-radio-desc">
                    <?php _e('Separate tab in product data area.', 'smart-nutrition-labels'); ?>
                </div>
            </label>

            <label class="woonutri-radio-card <?php echo $position === 'both' ? 'checked' : ''; ?>">
                <input type="radio" name="woonutri_options[display_position]" value="both" <?php checked($position, 'both'); ?> />
                <div class="woonutri-radio-header">
                    <span class="woonutri-radio-check"></span>
                    <?php _e('Both Locations', 'smart-nutrition-labels'); ?>
                </div>
                <div class="woonutri-radio-desc">
                    <?php _e('Show in summary AND dedicated tab.', 'smart-nutrition-labels'); ?>
                </div>
            </label>
        </div>
        <?php
    }

    public static function tab_title_field()
    {
        $options = self::get_options();
        $title = isset($options['tab_title']) ? $options['tab_title'] : 'Nutrition Facts';
        ?>
        <div class="woonutri-tab-config woonutri-field-wrapper"
            style="<?php echo (isset($options['display_position']) && $options['display_position'] === 'summary') ? 'display: none;' : ''; ?>">
            <input type="text" name="woonutri_options[tab_title]" value="<?php echo esc_attr($title); ?>" maxlength="50"
                class="woonutri-input" style="max-width: 300px;" />
            <p class="woonutri-help-text">
                <?php _e('The title that will appear on the nutrition facts tab. Maximum 50 characters.', 'smart-nutrition-labels'); ?>
            </p>
        </div>
        <?php
    }

    public static function tab_priority_field()
    {
        $options = self::get_options();
        $priority = isset($options['tab_priority']) ? $options['tab_priority'] : 50;
        ?>
        <div class="woonutri-tab-config woonutri-field-wrapper"
            style="<?php echo (isset($options['display_position']) && $options['display_position'] === 'summary') ? 'display: none;' : ''; ?>">
            <input type="number" name="woonutri_options[tab_priority]" value="<?php echo esc_attr($priority); ?>" min="1"
                max="100" class="woonutri-input" style="max-width: 100px;" />
            <p class="woonutri-help-text">
                <?php _e('Tab display priority (1-100). Lower numbers appear first. Default WooCommerce tabs use priorities around 10-50.', 'smart-nutrition-labels'); ?>
            </p>
        </div>
        <?php
    }

    public static function tab_show_icon_field()
    {
        $options = self::get_options();
        $show_icon = isset($options['tab_show_icon']) ? (bool) $options['tab_show_icon'] : true;
        ?>
        <div class="woonutri-tab-config"
            style="<?php echo (isset($options['display_position']) && $options['display_position'] === 'summary') ? 'display: none;' : ''; ?>">
            <label class="woonutri-toggle">
                <input type="checkbox" name="woonutri_options[tab_show_icon]" value="1" <?php checked($show_icon, true); ?> />
                <span class="woonutri-slider"></span>
            </label>
            <p class="description woonutri-help-text" style="display: inline-block; margin-left: 10px; vertical-align: top;">
                <?php _e('Show nutrition icon in tab title', 'smart-nutrition-labels'); ?>
            </p>
        </div>
        <?php
    }
    public static function label_theme_field()
    {
        $options = self::get_options();
        $theme = isset($options['label_theme']) ? $options['label_theme'] : 'basic';
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();

        
        $free_themes = ['basic', 'minimal', 'clean', 'simple', 'compact'];

        if (!$is_pro) {
            echo '<input type="hidden" name="woonutri_options[label_theme]" value="basic" />';
        } else {
            
            echo '<input type="hidden" id="woonutri-theme-input" name="woonutri_options[label_theme]" value="' . esc_attr($theme) . '" />';
        }

        
        $free_themes_list = [
            'basic' => 'WooNutri',
            'minimal' => 'Minimal',
            'clean' => 'Clean',
            'simple' => 'Simple',
            'compact' => 'Compact',
        ];

        $regional_themes = [
            'fda' => ['label' => 'FDA', 'flag' => '🇺🇸'],
            'eu' => ['label' => 'EU', 'flag' => '🇪🇺'],
            'uk' => ['label' => 'UK', 'flag' => '🇬🇧'],
            'australia' => ['label' => 'Australia', 'flag' => '🇦🇺'],
            'canada' => ['label' => 'Canada', 'flag' => '🇨🇦'],
            'nutriscore' => ['label' => 'Nutri-Score', 'flag' => '🇫🇷'],
            'latin' => ['label' => 'Latin', 'flag' => '🇲🇽'],
            'gda' => ['label' => 'GDA', 'flag' => '📊'],
        ];

        $design_themes = [
            'modern' => 'Modern',
            'dark' => 'Dark',
            'premium' => 'Premium',
            'corporate' => 'Corporate',
            'spacious' => 'Spacious',
            'boxed' => 'Boxed',
            'outline' => 'Outline',
            'gradient' => 'Gradient',
            'glass' => 'Glass',
            'paper' => 'Paper',
        ];

        $specialty_themes = [
            'organic' => ['label' => 'Organic', 'icon' => '🌿'],
            'sport' => ['label' => 'Sport', 'icon' => '💪'],
            'kids' => ['label' => 'Kids', 'icon' => '🎨'],
            'vegan' => ['label' => 'Vegan', 'icon' => '🌱'],
            'keto' => ['label' => 'Keto', 'icon' => '🥑'],
            'medical' => ['label' => 'Medical', 'icon' => '🏥'],
            'artisan' => ['label' => 'Artisan', 'icon' => '🎨'],
            'eco' => ['label' => 'Eco', 'icon' => '♻️'],
            'luxury' => ['label' => 'Luxury', 'icon' => '✨'],
        ];

        $creative_themes = [
            'retro' => ['label' => 'Retro', 'icon' => '📻'],
            'tech' => ['label' => 'Tech', 'icon' => '💻'],
            'neon' => ['label' => 'Neon', 'icon' => '💜'],
            'classic' => ['label' => 'Classic', 'icon' => '📜'],
        ];
        ?>
        <div class="woonutri-theme-selector woonutri-theme-compact">
            <!-- FREE Themes -->
            <div class="woonutri-theme-group">
                <h4 class="woonutri-theme-group-title-compact"><?php _e('FREE', 'smart-nutrition-labels'); ?></h4>
                <div class="woonutri-theme-chips">
                    <?php foreach ($free_themes_list as $key => $label):
                        $is_active = $theme === $key;
                    ?>
                        <button type="button" class="woonutri-theme-chip <?php echo $is_active ? 'active' : ''; ?>" data-theme="<?php echo esc_attr($key); ?>">
                            <?php echo esc_html($label); ?>
                        </button>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- REGIONAL Themes -->
            <div class="woonutri-theme-group">
                <h4 class="woonutri-theme-group-title-compact"><?php _e('REGIONAL', 'smart-nutrition-labels'); ?> <?php if (!$is_pro): ?><span class="woonutri-pro-badge">Pro</span><?php endif; ?></h4>
                <div class="woonutri-theme-chips">
                    <?php foreach ($regional_themes as $key => $data):
                        $is_active = $theme === $key;
                        $is_locked = !$is_pro;
                    ?>
                        <button type="button" class="woonutri-theme-chip <?php echo $is_active ? 'active' : ''; ?> <?php echo $is_locked ? 'locked' : ''; ?>" data-theme="<?php echo esc_attr($key); ?>">
                            <?php echo $data['flag']; ?> <?php echo esc_html($data['label']); ?><?php if ($is_locked): ?> 🔒<?php endif; ?>
                        </button>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- DESIGN STYLES -->
            <div class="woonutri-theme-group">
                <h4 class="woonutri-theme-group-title-compact"><?php _e('DESIGN STYLES', 'smart-nutrition-labels'); ?> <?php if (!$is_pro): ?><span class="woonutri-pro-badge">Pro</span><?php endif; ?></h4>
                <div class="woonutri-theme-chips">
                    <?php foreach ($design_themes as $key => $label):
                        $is_active = $theme === $key;
                        $is_locked = !$is_pro;
                    ?>
                        <button type="button" class="woonutri-theme-chip <?php echo $is_active ? 'active' : ''; ?> <?php echo $is_locked ? 'locked' : ''; ?>" data-theme="<?php echo esc_attr($key); ?>">
                            <?php echo esc_html($label); ?><?php if ($is_locked): ?> 🔒<?php endif; ?>
                        </button>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- SPECIALTY -->
            <div class="woonutri-theme-group">
                <h4 class="woonutri-theme-group-title-compact"><?php _e('SPECIALTY', 'smart-nutrition-labels'); ?> <?php if (!$is_pro): ?><span class="woonutri-pro-badge">Pro</span><?php endif; ?></h4>
                <div class="woonutri-theme-chips">
                    <?php foreach ($specialty_themes as $key => $data):
                        $is_active = $theme === $key;
                        $is_locked = !$is_pro;
                    ?>
                        <button type="button" class="woonutri-theme-chip <?php echo $is_active ? 'active' : ''; ?> <?php echo $is_locked ? 'locked' : ''; ?>" data-theme="<?php echo esc_attr($key); ?>">
                            <?php echo $data['icon']; ?> <?php echo esc_html($data['label']); ?><?php if ($is_locked): ?> 🔒<?php endif; ?>
                        </button>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- CREATIVE -->
            <div class="woonutri-theme-group">
                <h4 class="woonutri-theme-group-title-compact"><?php _e('CREATIVE', 'smart-nutrition-labels'); ?> <?php if (!$is_pro): ?><span class="woonutri-pro-badge">Pro</span><?php endif; ?></h4>
                <div class="woonutri-theme-chips">
                    <?php foreach ($creative_themes as $key => $data):
                        $is_active = $theme === $key;
                        $is_locked = !$is_pro;
                    ?>
                        <button type="button" class="woonutri-theme-chip <?php echo $is_active ? 'active' : ''; ?> <?php echo $is_locked ? 'locked' : ''; ?>" data-theme="<?php echo esc_attr($key); ?>">
                            <?php echo $data['icon']; ?> <?php echo esc_html($data['label']); ?><?php if ($is_locked): ?> 🔒<?php endif; ?>
                        </button>
                    <?php endforeach; ?>
                </div>
            </div>

            <?php if (!$is_pro): ?>
            <!-- Upgrade CTA -->
            <div class="woonutri-upgrade-cta-compact">
                <a href="<?php echo esc_url(woonutri_fs()->get_upgrade_url()); ?>" class="woonutri-upgrade-link">
                    🚀 <?php _e('Unlock all 35+ themes', 'smart-nutrition-labels'); ?>
                </a>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    public static function primary_color_field()
    {
        $options = self::get_options();
        $color = isset($options['primary_color']) ? $options['primary_color'] : '#2a7ae2';
        echo '<input type="text" class="woonutri-color-picker" name="woonutri_options[primary_color]" value="' . esc_attr($color) . '" data-default-color="#2a7ae2" />';
    }

    public static function font_family_field()
    {
        $options = self::get_options();
        $font = isset($options['font_family']) ? $options['font_family'] : 'default';
        $fonts = [
            'default' => __('Default', 'smart-nutrition-labels'),
            'sans' => __('Sans-serif', 'smart-nutrition-labels'),
            'serif' => __('Serif', 'smart-nutrition-labels'),
            'mono' => __('Monospace', 'smart-nutrition-labels')
        ];
        ?>
        <div class="woonutri-radio-group">
            <?php foreach ($fonts as $key => $label): ?>
                <label class="woonutri-radio-card <?php echo $font === $key ? 'checked' : ''; ?>" style="min-width: 100px;">
                    <input type="radio" name="woonutri_options[font_family]" value="<?php echo esc_attr($key); ?>" <?php checked($font, $key); ?> />
                    <div class="woonutri-radio-header" style="justify-content: center;">
                        <?php echo esc_html($label); ?>
                    </div>
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }

    public static function custom_font_field()
    {
        $options = self::get_options();
        $font = isset($options['custom_font']) ? $options['custom_font'] : '';
        $fonts = [
            '' => __('Default', 'smart-nutrition-labels'),
            'Roboto' => 'Roboto',
            'Open Sans' => 'Open Sans',
            'Lato' => 'Lato',
            'Montserrat' => 'Montserrat',
            'Merriweather' => 'Merriweather',
            'Oswald' => 'Oswald',
            'Raleway' => 'Raleway',
            'Nunito' => 'Nunito',
            'Poppins' => 'Poppins',
            'Source Sans Pro' => 'Source Sans Pro',
        ];
        echo '<div class="woonutri-field-wrapper">';
        echo '<select name="woonutri_options[custom_font]" id="woonutri_custom_font" class="woonutri-input" style="max-width: 300px;">';
        foreach ($fonts as $val => $label) {
            echo '<option value="' . esc_attr($val) . '"' . selected($font, $val, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo ' <a href="https://fonts.google.com/" target="_blank" class="button" style="vertical-align: top;">' . __('Preview fonts', 'smart-nutrition-labels') . '</a>';
        echo '</div>';
    }

    public static function font_size_field()
    {
        $options = self::get_options();
        $size = isset($options['font_size']) ? $options['font_size'] : 'normal';
        $sizes = [
            'small' => __('Small', 'smart-nutrition-labels'),
            'normal' => __('Normal', 'smart-nutrition-labels'),
            'large' => __('Large', 'smart-nutrition-labels'),
            'custom' => __('Custom', 'smart-nutrition-labels')
        ];
        ?>
        <div class="woonutri-radio-group woonutri-font-size-group">
            <?php foreach ($sizes as $key => $label): ?>
                <label class="woonutri-radio-card <?php echo $size === $key ? 'checked' : ''; ?>" style="min-width: 80px;">
                    <input type="radio" name="woonutri_options[font_size]" value="<?php echo esc_attr($key); ?>" <?php checked($size, $key); ?> />
                    <div class="woonutri-radio-header" style="justify-content: center;">
                        <?php echo esc_html($label); ?>
                    </div>
                </label>
            <?php endforeach; ?>
        </div>
        <div class="woonutri-custom-font-size"
            style="margin-top: 10px; <?php echo $size !== 'custom' ? 'display:none;' : ''; ?>">
            <input type="number" min="10" max="32" step="1" name="woonutri_options[font_size_custom]"
                value="<?php echo esc_attr(isset($options['font_size_custom']) ? $options['font_size_custom'] : '16'); ?>"
                class="woonutri-input" style="width:80px;" /> px
        </div>
        <?php
    }
    public static function table_layout_field()
    {
        $options = self::get_options();
        $layout = isset($options['table_layout']) ? $options['table_layout'] : 'standard';
        $layouts = [
            'compact' => __('Compact', 'smart-nutrition-labels'),
            'standard' => __('Standard', 'smart-nutrition-labels'),
            'spacious' => __('Spacious', 'smart-nutrition-labels')
        ];
        ?>
        <div class="woonutri-radio-group">
            <?php foreach ($layouts as $key => $label): ?>
                <label class="woonutri-radio-card <?php echo $layout === $key ? 'checked' : ''; ?>" style="min-width: 100px;">
                    <input type="radio" name="woonutri_options[table_layout]" value="<?php echo esc_attr($key); ?>" <?php checked($layout, $key); ?> />
                    <div class="woonutri-radio-header" style="justify-content: center;">
                        <?php echo esc_html($label); ?>
                    </div>
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }
    public static function unit_field($key)
    {
        $options = self::get_options();
        $val = isset($options['unit_' . $key]) ? $options['unit_' . $key] : ($key === 'energy_kcal' ? 'kcal' : 'g');
        echo '<input type="text" name="woonutri_options[unit_' . esc_attr($key) . ']" value="' . esc_attr($val) . '" class="woonutri-input" style="width:60px;" />';
    }

    public static function background_color_field()
    {
        $options = self::get_options();
        $color = isset($options['background_color']) ? $options['background_color'] : '#fff';
        echo '<input type="text" class="woonutri-color-picker" name="woonutri_options[background_color]" value="' . esc_attr($color) . '" data-default-color="#ffffff" />';
    }
    public static function text_color_field()
    {
        $options = self::get_options();
        $color = isset($options['text_color']) ? $options['text_color'] : '#222';
        echo '<input type="text" class="woonutri-color-picker" name="woonutri_options[text_color]" value="' . esc_attr($color) . '" data-default-color="#222222" />';
    }
    public static function header_color_field()
    {
        $options = self::get_options();
        $color = isset($options['header_color']) ? $options['header_color'] : '#2a7ae2';
        echo '<input type="text" class="woonutri-color-picker" name="woonutri_options[header_color]" value="' . esc_attr($color) . '" data-default-color="#2a7ae2" />';
    }
    public static function border_color_field()
    {
        $options = self::get_options();
        $color = isset($options['border_color']) ? $options['border_color'] : '#222';
        echo '<input type="text" class="woonutri-color-picker" name="woonutri_options[border_color]" value="' . esc_attr($color) . '" data-default-color="#222222" />';
    }

    public static function border_radius_field()
    {
        $options = self::get_options();
        $radius = isset($options['border_radius']) ? $options['border_radius'] : 'medium';
        echo '<select name="woonutri_options[border_radius]">';
        echo '<option value="none"' . selected($radius, 'none', false) . '>' . __('None', 'smart-nutrition-labels') . '</option>';
        echo '<option value="small"' . selected($radius, 'small', false) . '>' . __('Small', 'smart-nutrition-labels') . '</option>';
        echo '<option value="medium"' . selected($radius, 'medium', false) . '>' . __('Medium', 'smart-nutrition-labels') . '</option>';
        echo '<option value="large"' . selected($radius, 'large', false) . '>' . __('Large', 'smart-nutrition-labels') . '</option>';
        echo '</select>';
    }

    public static function show_logo_field()
    {
        $options = self::get_options();
        $show = isset($options['show_logo']) ? (bool) $options['show_logo'] : false;
        echo '<input type="checkbox" name="woonutri_options[show_logo]" value="1"' . checked($show, true, false) . ' /> ' . __('Display the Woonutri logo on the label', 'smart-nutrition-labels');
    }

    public static function enable_collapsible_field()
    {
        $options = self::get_options();
        $enable = isset($options['enable_collapsible']) ? (bool) $options['enable_collapsible'] : true;
        echo '<input type="checkbox" name="woonutri_options[enable_collapsible]" value="1"' . checked($enable, true, false) . ' /> ' . __('Enable collapsible sections for detailed nutrition information', 'smart-nutrition-labels');
        echo '<p class="description">' . __('When enabled, detailed nutrition information will be hidden by default and can be expanded by clicking.', 'smart-nutrition-labels') . '</p>';
    }

    public static function pro_styling_section_callback()
    {
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        if (!$is_pro) {
            echo '<div class="woonutri-pro-section-notice">';
            echo '<span class="dashicons dashicons-lock"></span> ';
            echo __('These advanced styling options are available with WooNutri Pro.', 'smart-nutrition-labels');
            echo ' <a href="' . esc_url(woonutri_fs()->get_upgrade_url()) . '">' . __('Upgrade Now', 'smart-nutrition-labels') . '</a>';
            echo '</div>';
        }
    }

    public static function enable_shadow_field()
    {
        $options = self::get_options();
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        $enable = isset($options['enable_shadow']) ? (bool) $options['enable_shadow'] : false;
        $disabled = $is_pro ? '' : 'disabled';
        echo '<input type="checkbox" name="woonutri_options[enable_shadow]" value="1"' . checked($enable, true, false) . ' ' . $disabled . ' /> ';
        echo __('Add shadow effect to nutrition labels', 'smart-nutrition-labels');
    }

    public static function shadow_intensity_field()
    {
        $options = self::get_options();
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        $intensity = isset($options['shadow_intensity']) ? $options['shadow_intensity'] : 'medium';
        $disabled = $is_pro ? '' : 'disabled';
        ?>
        <select name="woonutri_options[shadow_intensity]" <?php echo $disabled; ?>>
            <option value="light" <?php selected($intensity, 'light'); ?>><?php _e('Light', 'smart-nutrition-labels'); ?></option>
            <option value="medium" <?php selected($intensity, 'medium'); ?>><?php _e('Medium', 'smart-nutrition-labels'); ?></option>
            <option value="strong" <?php selected($intensity, 'strong'); ?>><?php _e('Strong', 'smart-nutrition-labels'); ?></option>
        </select>
        <?php
    }

    public static function enable_gradient_field()
    {
        $options = self::get_options();
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        $enable = isset($options['enable_gradient']) ? (bool) $options['enable_gradient'] : false;
        $disabled = $is_pro ? '' : 'disabled';
        echo '<input type="checkbox" name="woonutri_options[enable_gradient]" value="1"' . checked($enable, true, false) . ' ' . $disabled . ' /> ';
        echo __('Use gradient background for labels', 'smart-nutrition-labels');
    }

    public static function gradient_colors_field()
    {
        $options = self::get_options();
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        $color1 = isset($options['gradient_color_1']) ? $options['gradient_color_1'] : '#ffffff';
        $color2 = isset($options['gradient_color_2']) ? $options['gradient_color_2'] : '#f8fafc';
        $disabled = $is_pro ? '' : 'disabled';
        ?>
        <div style="display: flex; gap: 15px; align-items: center;">
            <div>
                <label><?php _e('Start', 'smart-nutrition-labels'); ?></label>
                <input type="color" name="woonutri_options[gradient_color_1]" value="<?php echo esc_attr($color1); ?>" <?php echo $disabled; ?>>
            </div>
            <div>
                <label><?php _e('End', 'smart-nutrition-labels'); ?></label>
                <input type="color" name="woonutri_options[gradient_color_2]" value="<?php echo esc_attr($color2); ?>" <?php echo $disabled; ?>>
            </div>
        </div>
        <?php
    }

    public static function enable_animations_field()
    {
        $options = self::get_options();
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        $enable = isset($options['enable_animations']) ? (bool) $options['enable_animations'] : false;
        $disabled = $is_pro ? '' : 'disabled';
        echo '<input type="checkbox" name="woonutri_options[enable_animations]" value="1"' . checked($enable, true, false) . ' ' . $disabled . ' /> ';
        echo __('Enable hover animations on labels', 'smart-nutrition-labels');
        echo '<p class="description">' . __('Adds subtle hover effects like scale and shadow transitions.', 'smart-nutrition-labels') . '</p>';
    }

    public static function custom_css_field()
    {
        $options = self::get_options();
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        $css = isset($options['custom_css']) ? $options['custom_css'] : '';
        $disabled = $is_pro ? '' : 'disabled';
        ?>
        <textarea name="woonutri_options[custom_css]" rows="6" style="width: 100%; max-width: 500px; font-family: monospace;"
            <?php echo $disabled; ?>
            placeholder="<?php echo esc_attr('.woonutri-label { /* your styles */ }'); ?>"><?php echo esc_textarea($css); ?></textarea>
        <p class="description"><?php _e('Add custom CSS to further customize the label appearance.', 'smart-nutrition-labels'); ?></p>
        <?php
    }

    
    public static function features_section_callback()
    {
        echo '<p class="description">' . __('Enable or disable optional features for nutrition labels.', 'smart-nutrition-labels') . '</p>';
    }

    
    public static function show_pdf_button_field()
    {
        $options = self::get_options();
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        $enabled = isset($options['show_pdf_button']) ? (bool) $options['show_pdf_button'] : false;
        $disabled = $is_pro ? '' : 'disabled';
        ?>
        <label class="woonutri-toggle">
            <input type="checkbox" name="woonutri_options[show_pdf_button]" value="1" <?php checked($enabled, true); ?> <?php echo $disabled; ?> />
            <span class="woonutri-slider"></span>
        </label>
        <span style="margin-left: 10px;"><?php _e('Show PDF download button on nutrition labels', 'smart-nutrition-labels'); ?></span>
        <?php if (!$is_pro): ?>
            <p class="description" style="color: #996800;"><span class="dashicons dashicons-lock"></span> <?php _e('PDF Export is a Pro feature.', 'smart-nutrition-labels'); ?></p>
        <?php endif; ?>
        <?php
    }

    
    public static function show_nutri_score_field()
    {
        $options = self::get_options();
        $enabled = isset($options['show_nutri_score']) ? (bool) $options['show_nutri_score'] : true;
        ?>
        <label class="woonutri-toggle">
            <input type="checkbox" name="woonutri_options[show_nutri_score]" value="1" <?php checked($enabled, true); ?> />
            <span class="woonutri-slider"></span>
        </label>
        <span style="margin-left: 10px;"><?php _e('Display Nutri-Score badge on labels (when calculated)', 'smart-nutrition-labels'); ?></span>
        <p class="description"><?php _e('Nutri-Score is automatically calculated when nutrition data is saved.', 'smart-nutrition-labels'); ?></p>
        <?php
    }

    
    public static function enable_schema_seo_field()
    {
        $options = self::get_options();
        $enabled = isset($options['enable_schema_seo']) ? (bool) $options['enable_schema_seo'] : true;
        ?>
        <label class="woonutri-toggle">
            <input type="checkbox" name="woonutri_options[enable_schema_seo]" value="1" <?php checked($enabled, true); ?> />
            <span class="woonutri-slider"></span>
        </label>
        <span style="margin-left: 10px;"><?php _e('Output Schema.org structured data for SEO', 'smart-nutrition-labels'); ?></span>
        <p class="description"><?php _e('Adds JSON-LD nutrition data to product pages for better search engine visibility.', 'smart-nutrition-labels'); ?></p>
        <?php
    }

    
    public static function display_format_field()
    {
        $options = self::get_options();
        $format = isset($options['display_format']) ? $options['display_format'] : 'per_100g';
        ?>
        <div class="woonutri-radio-group">
            <label class="woonutri-radio-card <?php echo $format === 'per_100g' ? 'checked' : ''; ?>" style="min-width: 100px;">
                <input type="radio" name="woonutri_options[display_format]" value="per_100g" <?php checked($format, 'per_100g'); ?> />
                <div class="woonutri-radio-header" style="justify-content: center;">
                    <?php _e('Per 100g', 'smart-nutrition-labels'); ?>
                </div>
            </label>
            <label class="woonutri-radio-card <?php echo $format === 'per_serving' ? 'checked' : ''; ?>" style="min-width: 100px;">
                <input type="radio" name="woonutri_options[display_format]" value="per_serving" <?php checked($format, 'per_serving'); ?> />
                <div class="woonutri-radio-header" style="justify-content: center;">
                    <?php _e('Per Serving', 'smart-nutrition-labels'); ?>
                </div>
            </label>
            <label class="woonutri-radio-card <?php echo $format === 'both' ? 'checked' : ''; ?>" style="min-width: 100px;">
                <input type="radio" name="woonutri_options[display_format]" value="both" <?php checked($format, 'both'); ?> />
                <div class="woonutri-radio-header" style="justify-content: center;">
                    <?php _e('Both', 'smart-nutrition-labels'); ?>
                </div>
            </label>
        </div>
        <p class="description"><?php _e('How to display nutrition values on the frontend label.', 'smart-nutrition-labels'); ?></p>
        <?php
    }

    
    public static function show_traffic_lights_field()
    {
        $options = self::get_options();
        $enabled = isset($options['show_traffic_lights']) ? (bool) $options['show_traffic_lights'] : false;
        $is_pro = function_exists('woonutri_is_pro') && woonutri_is_pro();
        ?>
        <label class="woonutri-toggle <?php echo !$is_pro ? 'woonutri-pro-locked' : ''; ?>">
            <input type="checkbox" name="woonutri_options[show_traffic_lights]" value="1"
                <?php checked($enabled, true); ?>
                <?php disabled(!$is_pro, true); ?> />
            <span class="woonutri-slider"></span>
        </label>
        <span style="margin-left: 10px;">
            <?php _e('Show UK Traffic Light color indicators (red/amber/green)', 'smart-nutrition-labels'); ?>
            <?php if (!$is_pro): ?>
                <span class="woonutri-pro-badge">Pro</span>
            <?php endif; ?>
        </span>
        <p class="description">
            <?php _e('Display color-coded indicators for fat, saturated fat, sugar, and salt based on UK FSA guidelines.', 'smart-nutrition-labels'); ?>
        </p>
        <?php
    }

    public static function labels_section_callback()
    {
        echo '<p class="description">' . __('Customize the text displayed on nutrition labels. Leave empty to use default translations.', 'smart-nutrition-labels') . '</p>';
    }

    public static function label_title_field()
    {
        $options = self::get_options();
        $value = isset($options['label_title']) ? $options['label_title'] : '';
        ?>
        <input type="text" name="woonutri_options[label_title]" value="<?php echo esc_attr($value); ?>" style="width: 300px;"
            placeholder="<?php echo esc_attr__('Nutrition Facts', 'smart-nutrition-labels'); ?>" />
        <p class="description"><?php _e('Override the main label title. Default: "Nutrition Facts"', 'smart-nutrition-labels'); ?></p>
        <?php
    }

    public static function label_per_serving_field()
    {
        $options = self::get_options();
        $value = isset($options['label_per_serving']) ? $options['label_per_serving'] : '';
        ?>
        <input type="text" name="woonutri_options[label_per_serving]" value="<?php echo esc_attr($value); ?>"
            style="width: 300px;" placeholder="<?php echo esc_attr__('Per serving', 'smart-nutrition-labels'); ?>" />
        <p class="description"><?php _e('Override "Per serving" text. Default: "Per serving"', 'smart-nutrition-labels'); ?></p>
        <?php
    }

    public static function label_per_100g_field()
    {
        $options = self::get_options();
        $value = isset($options['label_per_100g']) ? $options['label_per_100g'] : '';
        ?>
        <input type="text" name="woonutri_options[label_per_100g]" value="<?php echo esc_attr($value); ?>" style="width: 300px;"
            placeholder="<?php echo esc_attr__('Per 100g', 'smart-nutrition-labels'); ?>" />
        <p class="description"><?php _e('Override "Per 100g" text. Default: "Per 100g"', 'smart-nutrition-labels'); ?></p>
        <?php
    }

    public static function label_daily_value_field()
    {
        $options = self::get_options();
        $value = isset($options['label_daily_value']) ? $options['label_daily_value'] : '';
        ?>
        <input type="text" name="woonutri_options[label_daily_value]" value="<?php echo esc_attr($value); ?>"
            style="width: 300px;" placeholder="<?php echo esc_attr__('% Daily Value*', 'smart-nutrition-labels'); ?>" />
        <p class="description"><?php _e('Override daily value header text. Default: "% Daily Value*"', 'smart-nutrition-labels'); ?></p>
        <?php
    }

    public static function render_settings_page()
    {
        ?>
        <div class="wrap woonutri-settings-wrap">
            <div class="woonutri-settings-header">
                <h1><?php _e('WooNutri Settings', 'smart-nutrition-labels'); ?></h1>
                <p><?php _e('Configure how nutrition labels appear on your WooCommerce products.', 'smart-nutrition-labels'); ?></p>
            </div>

            <?php
            
            $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
            $notice_dismissed = get_user_meta(get_current_user_id(), 'woonutri_upgrade_notice_dismissed_until', true);
            $show_notice = !$is_pro && (!$notice_dismissed || $notice_dismissed < time());

            if ($show_notice):
                ?>
                <div class="notice notice-info is-dismissible woonutri-upgrade-notice" id="woonutri-upgrade-notice">
                    <div class="woonutri-upgrade-content">
                        <div class="woonutri-upgrade-icon">⭐</div>
                        <div class="woonutri-upgrade-text">
                            <h3><?php _e('Upgrade to WooNutri Pro', 'smart-nutrition-labels'); ?></h3>
                            <p><?php _e('Get access to FDA-compliant themes, Smart Automation (Allergen Boldifier & %DV Calculator), barcode lookup, and bulk operations!', 'smart-nutrition-labels'); ?>
                            </p>
                        </div>
                        <div class="woonutri-upgrade-actions">
                            <a href="<?php echo function_exists('woonutri_fs') ? woonutri_fs()->get_upgrade_url() : '#'; ?>"
                                class="button button-primary">
                                <?php _e('Upgrade Now', 'smart-nutrition-labels'); ?>
                            </a>
                            <div class="woonutri-dismiss-options">
                                <button type="button" class="button woonutri-dismiss-notice" data-period="week"
                                    data-nonce="<?php echo wp_create_nonce('woonutri_dismiss_notice'); ?>">
                                    <?php _e('Remind me in 1 week', 'smart-nutrition-labels'); ?>
                                </button>
                                <button type="button" class="button woonutri-dismiss-notice" data-period="month"
                                    data-nonce="<?php echo wp_create_nonce('woonutri_dismiss_notice'); ?>">
                                    <?php _e('Remind me in 1 month', 'smart-nutrition-labels'); ?>
                                </button>
                                <button type="button" class="button woonutri-dismiss-notice" data-period="forever"
                                    data-nonce="<?php echo wp_create_nonce('woonutri_dismiss_notice'); ?>">
                                    <?php _e('Don\'t show again', 'smart-nutrition-labels'); ?>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <style>
                .woonutri-upgrade-notice {
                    background: #fff;
                    border: 1px solid #c3c4c7;
                    border-left: 4px solid #2563eb;
                    margin: 0 0 20px 0;
                    padding: 12px;
                    box-shadow: 0 1px 1px rgba(0, 0, 0, .04);
                }

                .woonutri-upgrade-content {
                    display: flex;
                    align-items: flex-start;
                    gap: 12px;
                }

                .woonutri-upgrade-icon {
                    font-size: 20px;
                    margin-top: 2px;
                    flex-shrink: 0;
                }

                .woonutri-upgrade-text {
                    flex: 1;
                }

                .woonutri-upgrade-text h3 {
                    margin: 0 0 4px 0;
                    font-size: 14px;
                    line-height: 1.4;
                    font-weight: 600;
                    color: #1d2327;
                }

                .woonutri-upgrade-text p {
                    margin: 0 0 8px 0;
                    font-size: 13px;
                    line-height: 1.5;
                    color: #646970;
                }

                .woonutri-upgrade-actions {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 8px;
                    align-items: center;
                }

                .woonutri-dismiss-options {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 6px;
                    margin-left: 8px;
                }

                .woonutri-dismiss-options .button {
                    font-size: 12px;
                    height: 28px;
                    line-height: 26px;
                    padding: 0 8px;
                    border-color: #c3c4c7;
                    color: #646970;
                }

                .woonutri-dismiss-options .button:hover {
                    border-color: #8c8f94;
                    color: #1d2327;
                }

                /* WordPress admin notice compliance */
                .woonutri-upgrade-notice.notice {
                    position: relative;
                }

                .woonutri-upgrade-notice .notice-dismiss {
                    display: none;
                }

                @media screen and (max-width: 782px) {
                    .woonutri-upgrade-content {
                        flex-direction: column;
                        gap: 8px;
                    }

                    .woonutri-upgrade-actions {
                        justify-content: flex-start;
                    }

                    .woonutri-dismiss-options {
                        margin-left: 0;
                        margin-top: 4px;
                    }
                }
            </style>

            <div class="woonutri-settings-content">
                <div class="woonutri-settings-main">
                    <!-- Modern Tab Navigation -->
                    <div class="woonutri-modern-tabs">
                        <div class="woonutri-tab-buttons">
                            <button type="button" class="woonutri-tab-btn active" data-tab="display">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                                </svg>
                                <?php _e('Display', 'smart-nutrition-labels'); ?>
                            </button>
                            <button type="button" class="woonutri-tab-btn" data-tab="appearance">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M7 21a4 4 0 01-4-4V5a2 2 0 012-2h4a2 2 0 012 2v12a4 4 0 01-4 4zM21 5a2 2 0 00-2-2h-4a2 2 0 00-2 2v12a4 4 0 004 4h4a2 2 0 002-2V5z" />
                                </svg>
                                <?php _e('Appearance', 'smart-nutrition-labels'); ?>
                            </button>
                            <button type="button" class="woonutri-tab-btn" data-tab="advanced">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M12 6V4m0 2a2 2 0 100 4m0-4a2 2 0 110 4m-6 8a2 2 0 100-4m0 4a2 2 0 100 4m0-4v2m0-6V4m6 6v10m6-2a2 2 0 100-4m0 4a2 2 0 100 4m0-4v2m0-6V4" />
                                </svg>
                                <?php _e('Advanced', 'smart-nutrition-labels'); ?>
                            </button>
                            <?php if (function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code()): ?>
                                <button type="button" class="woonutri-tab-btn" data-tab="pro">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M11.049 2.927c.3-.921 1.603-.921 1.902 0l1.519 4.674a1 1 0 00.95.69h4.915c.969 0 1.371 1.24.588 1.81l-3.976 2.888a1 1 0 00-.363 1.118l1.518 4.674c.3.922-.755 1.688-1.538 1.118l-3.976-2.888a1 1 0 00-1.176 0l-3.976 2.888c-.783.57-1.838-.197-1.538-1.118l1.518-4.674a1 1 0 00-.363-1.118l-3.976-2.888c-.784-.57-.38-1.81.588-1.81h4.914a1 1 0 00.951-.69l1.519-4.674z" />
                                    </svg>
                                    <?php _e('Pro Features', 'smart-nutrition-labels'); ?>
                                </button>
                            <?php endif; ?>
                        </div>

                        <form method="post" action="options.php" id="woonutri-settings-form">
                            <?php settings_fields('woonutri_options'); ?>

                            <!-- Display Tab -->
                            <div class="woonutri-tab-panel active" data-panel="display">
                                <?php self::render_display_tab(); ?>
                            </div>

                            <!-- Appearance Tab -->
                            <div class="woonutri-tab-panel" data-panel="appearance">
                                <?php self::render_appearance_tab(); ?>
                            </div>

                            <!-- Advanced Tab -->
                            <div class="woonutri-tab-panel" data-panel="advanced">
                                <?php self::render_advanced_tab(); ?>
                            </div>

                            <?php if (function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code()): ?>
                                <!-- Pro Features Tab -->
                                <div class="woonutri-tab-panel" data-panel="pro">
                                    <?php self::render_pro_tab(); ?>
                                </div>
                            <?php endif; ?>

                            <div class="woonutri-form-actions">
                                <?php submit_button(__('Save Changes', 'smart-nutrition-labels'), 'primary', 'submit', false); ?>
                                <button type="button" class="button"
                                    id="woonutri-reset-btn"><?php _e('Reset to Default', 'smart-nutrition-labels'); ?></button>
                            </div>
                        </form>
                    </div>
                </div>

                <div class="woonutri-preview-sidebar">
                    <h2><?php _e('Live Label Preview', 'smart-nutrition-labels'); ?></h2>

                    <div id="woonutri-preview">
                        <?php echo self::sample_label_html(); ?>
                    </div>

                    <div id="woonutri-preview-controls">
                        <div class="woonutri-theme-preview-section">
                            <strong><?php _e('Theme Preview', 'smart-nutrition-labels'); ?>:</strong>
                            <p class="description" style="margin: 5px 0 10px; font-size: 12px; color: #666;">
                                <?php _e('35+ label themes for every market and style. Click to preview.', 'smart-nutrition-labels'); ?>
                            </p>

                            <!-- Free Themes -->
                            <div class="woonutri-theme-group">
                                <span class="woonutri-theme-group-label"><?php _e('Free', 'smart-nutrition-labels'); ?></span>
                                <div class="woonutri-theme-buttons">
                                    <button type="button" class="woonutri-theme-btn active" data-theme="basic"
                                        title="<?php esc_attr_e('WooNutri signature theme with customizable colors', 'smart-nutrition-labels'); ?>"><?php _e('WooNutri', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn" data-theme="minimal"
                                        title="<?php esc_attr_e('Minimalist design, no borders', 'smart-nutrition-labels'); ?>"><?php _e('Minimal', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn" data-theme="clean"
                                        title="<?php esc_attr_e('Clean design with subtle styling', 'smart-nutrition-labels'); ?>"><?php _e('Clean', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn" data-theme="simple"
                                        title="<?php esc_attr_e('Ultra-simple, back-to-basics style', 'smart-nutrition-labels'); ?>"><?php _e('Simple', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn" data-theme="compact"
                                        title="<?php esc_attr_e('Space-efficient condensed layout', 'smart-nutrition-labels'); ?>"><?php _e('Compact', 'smart-nutrition-labels'); ?></button>
                                </div>
                            </div>

                            <!-- Regional/Compliance Themes -->
                            <div class="woonutri-theme-group">
                                <span class="woonutri-theme-group-label"><?php _e('Regional', 'smart-nutrition-labels'); ?> <span class="dashicons dashicons-admin-site" style="font-size: 12px; width: 12px; height: 12px;"></span></span>
                                <div class="woonutri-theme-buttons">
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="fda"
                                        title="<?php esc_attr_e('US FDA 2020 Nutrition Facts format', 'smart-nutrition-labels'); ?>">🇺🇸 <?php _e('FDA', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="eu"
                                        title="<?php esc_attr_e('EU 1169/2011 with kJ/kcal', 'smart-nutrition-labels'); ?>">🇪🇺 <?php _e('EU', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="uk"
                                        title="<?php esc_attr_e('UK Traffic Light system', 'smart-nutrition-labels'); ?>">🇬🇧 <?php _e('UK', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="australia"
                                        title="<?php esc_attr_e('Australia Health Star Rating', 'smart-nutrition-labels'); ?>">🇦🇺 <?php _e('Australia', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="canada"
                                        title="<?php esc_attr_e('Canada bilingual EN/FR format', 'smart-nutrition-labels'); ?>">🇨🇦 <?php _e('Canada', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="nutriscore"
                                        title="<?php esc_attr_e('European Nutri-Score A-E rating', 'smart-nutrition-labels'); ?>">🇫🇷 <?php _e('Nutri-Score', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="latin"
                                        title="<?php esc_attr_e('Chile/Mexico octagon warning style', 'smart-nutrition-labels'); ?>">🇲🇽 <?php _e('Latin', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="gda"
                                        title="<?php esc_attr_e('Guideline Daily Amounts bubbles', 'smart-nutrition-labels'); ?>">📊 <?php _e('GDA', 'smart-nutrition-labels'); ?></button>
                                </div>
                            </div>

                            <!-- Design Style Themes -->
                            <div class="woonutri-theme-group">
                                <span class="woonutri-theme-group-label"><?php _e('Design Styles', 'smart-nutrition-labels'); ?> <span class="dashicons dashicons-art" style="font-size: 12px; width: 12px; height: 12px;"></span></span>
                                <div class="woonutri-theme-buttons">
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="modern"
                                        title="<?php esc_attr_e('Modern design with gradient header', 'smart-nutrition-labels'); ?>"><?php _e('Modern', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="dark"
                                        title="<?php esc_attr_e('Dark mode for dark backgrounds', 'smart-nutrition-labels'); ?>"><?php _e('Dark', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="premium"
                                        title="<?php esc_attr_e('Premium elegant serif design', 'smart-nutrition-labels'); ?>"><?php _e('Premium', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="corporate"
                                        title="<?php esc_attr_e('Professional corporate styling', 'smart-nutrition-labels'); ?>"><?php _e('Corporate', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="spacious"
                                        title="<?php esc_attr_e('Large, readable layout', 'smart-nutrition-labels'); ?>"><?php _e('Spacious', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="boxed"
                                        title="<?php esc_attr_e('Boxed sections with shadows', 'smart-nutrition-labels'); ?>"><?php _e('Boxed', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="outline"
                                        title="<?php esc_attr_e('Clean outlined borders', 'smart-nutrition-labels'); ?>"><?php _e('Outline', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="gradient"
                                        title="<?php esc_attr_e('Colorful gradient design', 'smart-nutrition-labels'); ?>"><?php _e('Gradient', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="glass"
                                        title="<?php esc_attr_e('Frosted glass effect', 'smart-nutrition-labels'); ?>"><?php _e('Glass', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="paper"
                                        title="<?php esc_attr_e('Printed document style', 'smart-nutrition-labels'); ?>"><?php _e('Paper', 'smart-nutrition-labels'); ?></button>
                                </div>
                            </div>

                            <!-- Specialty/Market Themes -->
                            <div class="woonutri-theme-group">
                                <span class="woonutri-theme-group-label"><?php _e('Specialty', 'smart-nutrition-labels'); ?> <span class="dashicons dashicons-star-filled" style="font-size: 12px; width: 12px; height: 12px;"></span></span>
                                <div class="woonutri-theme-buttons">
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="organic"
                                        title="<?php esc_attr_e('Natural & organic products', 'smart-nutrition-labels'); ?>">🌿 <?php _e('Organic', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="sport"
                                        title="<?php esc_attr_e('Sports nutrition & fitness', 'smart-nutrition-labels'); ?>">💪 <?php _e('Sport', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="kids"
                                        title="<?php esc_attr_e('Fun & colorful for kids', 'smart-nutrition-labels'); ?>">🎨 <?php _e('Kids', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="vegan"
                                        title="<?php esc_attr_e('Plant-based & vegan products', 'smart-nutrition-labels'); ?>">🌱 <?php _e('Vegan', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="keto"
                                        title="<?php esc_attr_e('Low-carb & keto focus', 'smart-nutrition-labels'); ?>">🥑 <?php _e('Keto', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="medical"
                                        title="<?php esc_attr_e('Clinical & pharmaceutical style', 'smart-nutrition-labels'); ?>">🏥 <?php _e('Medical', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="artisan"
                                        title="<?php esc_attr_e('Handcrafted artisan products', 'smart-nutrition-labels'); ?>">✋ <?php _e('Artisan', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="eco"
                                        title="<?php esc_attr_e('Eco-friendly sustainability', 'smart-nutrition-labels'); ?>">♻️ <?php _e('Eco', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="luxury"
                                        title="<?php esc_attr_e('Elegant luxury branding', 'smart-nutrition-labels'); ?>">✨ <?php _e('Luxury', 'smart-nutrition-labels'); ?></button>
                                </div>
                            </div>

                            <!-- Creative Themes -->
                            <div class="woonutri-theme-group">
                                <span class="woonutri-theme-group-label"><?php _e('Creative', 'smart-nutrition-labels'); ?> <span class="dashicons dashicons-palette" style="font-size: 12px; width: 12px; height: 12px;"></span></span>
                                <div class="woonutri-theme-buttons">
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="retro"
                                        title="<?php esc_attr_e('Vintage 1950s style', 'smart-nutrition-labels'); ?>">📻 <?php _e('Retro', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="tech"
                                        title="<?php esc_attr_e('Futuristic tech design', 'smart-nutrition-labels'); ?>">🔮 <?php _e('Tech', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="neon"
                                        title="<?php esc_attr_e('Vibrant neon colors', 'smart-nutrition-labels'); ?>">💜 <?php _e('Neon', 'smart-nutrition-labels'); ?></button>
                                    <button type="button" class="woonutri-theme-btn woonutri-theme-pro" data-theme="classic"
                                        title="<?php esc_attr_e('Traditional classic design', 'smart-nutrition-labels'); ?>"><?php _e('Classic', 'smart-nutrition-labels'); ?></button>
                                </div>
                            </div>
                        </div>

                        <strong><?php _e('Basic Fields', 'smart-nutrition-labels'); ?>:</strong>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 10px; margin-bottom: 15px;">
                            <label>Energy (kcal): <input type="number" id="woonutri_sample_energy" value="250" /></label>
                            <label>Fat (g): <input type="number" id="woonutri_sample_fat" value="12.5" step="0.1" /></label>
                            <label>Carbs (g): <input type="number" id="woonutri_sample_carbs" value="45.0" step="0.1" /></label>
                            <label>Sugar (g): <input type="number" id="woonutri_sample_sugar" value="15.2" step="0.1" /></label>
                            <label>Protein (g): <input type="number" id="woonutri_sample_protein" value="8.7"
                                    step="0.1" /></label>
                            <label>Fiber (g): <input type="number" id="woonutri_sample_fiber" value="3.2" step="0.1" /></label>
                            <label>Salt (g): <input type="number" id="woonutri_sample_salt" value="0.8" step="0.01" /></label>
                        </div>

                        <?php
                        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
                        if ($is_pro): ?>
                            <strong><?php _e('Pro Fields', 'smart-nutrition-labels'); ?>:</strong>
                            <div
                                style="display: grid; grid-template-columns: auto 1fr; gap: 10px; align-items: center; margin-bottom: 15px;">
                                <label><input type="checkbox" id="woonutri_toggle_cholesterol" checked /> Cholesterol:</label>
                                <input type="number" id="woonutri_sample_cholesterol" value="10" step="1" /> mg

                                <label><input type="checkbox" id="woonutri_toggle_trans_fat" checked /> Trans Fats:</label>
                                <input type="number" id="woonutri_sample_trans_fat" value="0.2" step="0.1" /> g

                                <label><input type="checkbox" id="woonutri_toggle_added_sugars" checked /> Added Sugars:</label>
                                <input type="number" id="woonutri_sample_added_sugars" value="1.5" step="0.1" /> g

                                <label><input type="checkbox" id="woonutri_toggle_vitamins" checked /> Vitamins:</label>
                                <input type="text" id="woonutri_sample_vitamins" value="Vit C: 20%, D: 10%" />

                                <label><input type="checkbox" id="woonutri_toggle_ingredients" checked /> Ingredients:</label>
                                <input type="text" id="woonutri_sample_ingredients" value="Whey protein, Cocoa, Natural flavors" />

                                <label><input type="checkbox" id="woonutri_toggle_barcode" checked /> Barcode:</label>
                                <input type="text" id="woonutri_sample_barcode" value="1234567890123" />
                            </div>

                            <strong><?php _e('Advanced Features', 'smart-nutrition-labels'); ?>:</strong>
                            <div style="display: grid; grid-template-columns: auto 1fr; gap: 10px; align-items: center;">
                                <label><input type="checkbox" id="woonutri_toggle_serving" checked /> Per Serving:</label>
                                <input type="text" id="woonutri_sample_serving" value="30g (1 bar)" />

                                <label><input type="checkbox" id="woonutri_toggle_dv" checked /> Daily Value %:</label>
                                <input type="checkbox" id="woonutri_show_dv" checked /> Show %DV
                            </div>
                        <?php else: ?>
                            <div class="woonutri-pro-preview-notice">
                                <p><strong><?php _e('🔒 Pro Features Preview', 'smart-nutrition-labels'); ?></strong></p>
                                <p class="description"><?php _e('Unlock advanced fields and features:', 'smart-nutrition-labels'); ?></p>
                                <ul style="list-style: none; padding: 0; margin: 10px 0;">
                                    <li>✓ <?php _e('Detailed nutrition breakdown', 'smart-nutrition-labels'); ?></li>
                                    <li>✓ <?php _e('Vitamins & minerals', 'smart-nutrition-labels'); ?></li>
                                    <li>✓ <?php _e('Allergen information', 'smart-nutrition-labels'); ?></li>
                                    <li>✓ <?php _e('AI-powered label scanning', 'smart-nutrition-labels'); ?></li>
                                    <li>✓ <?php _e('Multiple label themes', 'smart-nutrition-labels'); ?></li>
                                </ul>
                                <a href="<?php echo esc_url(woonutri_fs()->get_upgrade_url()); ?>" class="button button-primary">
                                    <?php _e('Upgrade to Pro', 'smart-nutrition-labels'); ?>
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    public static function sample_label_html()
    {
        if (!class_exists('Woonutri_Label_Renderer'))
            return '';

        
        $sample = [
            'energy_kcal' => '230',
            'energy_kj' => '963',
            'fat' => '8.0',
            'saturated_fat' => '1.0',
            'trans_fat' => '0.0',
            'cholesterol' => '0',
            'sodium' => '160',
            'carbs' => '37.0',
            'fiber' => '4.0',
            'sugar' => '12.0',
            'added_sugars' => '10.0',
            'protein' => '3.0',
            'salt' => '0.4',
            'potassium' => '240',
            'vitamin_d' => '2',
            'calcium' => '260',
            'iron' => '8',
            'serving_size' => '2/3 cup (55g)',
            'servings_per_container' => '8',
            'ingredients' => 'Mandarin Oranges (37.5%), Light Whipping Cream (Milk), Pears (12.5%), Seedless Grapes (7.6%), Apple (7.5%), Banana (5.9%)',
            'allergens' => 'Contains: Milk. May contain traces of nuts',
        ];

        return \Woonutri_Label_Renderer::render_label(0, $sample);
    }

    
    public static function enqueue_admin_scripts($hook)
    {
        
        echo "<script>console.log('Woonutri Debug: Current Admin Hook = \"' + '$hook' + '\"');</script>";
        
        
        
        $is_settings_page = (
            strpos($hook, 'woonutri-settings') !== false || 
            strpos($hook, 'woonutri_page_woonutri-settings') !== false ||
            (strpos($hook, 'smart-nutrition-labels') !== false && strpos($hook, 'settings') !== false) ||
            (isset($_GET['page']) && $_GET['page'] === 'woonutri-settings')
        );
        
        $is_features_page = strpos($hook, 'woonutri-features') !== false;

        
        if (!$is_settings_page && !$is_features_page) {
             
             if (strpos($hook, 'smart-nutrition-labels') === false && (!isset($_GET['page']) || strpos($_GET['page'], 'smart-nutrition-labels') === false)) {
                 return;
             }
        }

        
        wp_enqueue_style(
            'woonutri-admin-core',
            WOONUTRI_PLUGIN_URL . 'assets/css/admin-core.css',
            [],
            WOONUTRI_VERSION
        );

        if ($is_settings_page || $is_features_page) {
            wp_enqueue_style(
                'woonutri-admin-settings',
                WOONUTRI_PLUGIN_URL . 'assets/css/admin-settings.css',
                ['woonutri-admin-core'],
                WOONUTRI_VERSION . '.9' 
            );
        }

        
        wp_enqueue_style(
            'woonutri-label',
            WOONUTRI_PLUGIN_URL . 'assets/css/label.css',
            [],
            WOONUTRI_VERSION
        );

        wp_enqueue_style(
            'woonutri-label-fda',
            WOONUTRI_PLUGIN_URL . 'assets/css/label-fda.css',
            ['woonutri-label'],
            WOONUTRI_VERSION
        );

        wp_enqueue_style(
            'woonutri-label-eu',
            WOONUTRI_PLUGIN_URL . 'assets/css/label-eu.css',
            ['woonutri-label'],
            WOONUTRI_VERSION
        );

        
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        wp_enqueue_media();

        wp_enqueue_script(
            'woonutri-settings-js',
            WOONUTRI_PLUGIN_URL . 'assets/js/admin-settings.js',
            ['jquery', 'wp-color-picker', 'jquery-ui-tabs'],
            WOONUTRI_VERSION,
            true
        );

        
        wp_enqueue_script(
            'woonutri-label-preview',
            WOONUTRI_PLUGIN_URL . 'assets/js/label-preview.js',
            array('jquery'),
            WOONUTRI_VERSION,
            true
        );

        
        wp_localize_script('woonutri-settings-js', 'woonutri_settings', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('woonutri_get_preview')
        ]);

        
        wp_localize_script('woonutri-settings-js', 'woonutri_admin', [
            'clear_cache_nonce' => wp_create_nonce('woonutri_clear_cache')
        ]);

        
        wp_localize_script('woonutri-label-preview', 'woonutri_data', array(
            'plugin_url' => WOONUTRI_PLUGIN_URL,
            'is_pro' => function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code(),
            'nonce' => wp_create_nonce('woonutri_preview'),
            'ajax_url' => admin_url('admin-ajax.php')
        ));

        
        wp_add_inline_style('woonutri-admin-settings', '
            .woonutri-active-option {
                background-color: #f0f8ff;
                border-left: 3px solid #2a7ae2;
                padding-left: 10px;
                margin-left: -10px;
            }
            #woonutri-display-position label {
                display: block;
                margin-bottom: 8px;
                padding: 5px 0;
                transition: all 0.3s ease;
            }
            .woonutri-tab-config {
                transition: all 0.3s ease;
            }
        ');
    }

    
    public static function ajax_get_preview()
    {
        
        check_ajax_referer('woonutri_preview', 'nonce');

        if (!class_exists('Woonutri_Label_Renderer')) {
            wp_send_json_error('Label renderer not available');
        }

        
        $sample_data = isset($_POST['sample_data']) ? $_POST['sample_data'] : array();
        $settings = isset($_POST['settings']) ? $_POST['settings'] : array();

        
        $clean_sample = array();
        foreach ($sample_data as $key => $value) {
            $clean_sample[sanitize_key($key)] = sanitize_text_field($value);
        }

        
        $options = array();

        
        if (isset($settings['theme'])) {
            $options['theme'] = sanitize_text_field($settings['theme']);
        }

        
        if (isset($settings['primary_color'])) {
            $options['primary_color'] = sanitize_hex_color($settings['primary_color']);
        }
        if (isset($settings['background_color'])) {
            $options['background_color'] = sanitize_hex_color($settings['background_color']);
        }
        if (isset($settings['text_color'])) {
            $options['text_color'] = sanitize_hex_color($settings['text_color']);
        }
        if (isset($settings['header_color'])) {
            $options['header_color'] = sanitize_hex_color($settings['header_color']);
        }
        if (isset($settings['border_color'])) {
            $options['border_color'] = sanitize_hex_color($settings['border_color']);
        }

        
        if (isset($settings['font_family'])) {
            $options['font_family'] = sanitize_text_field($settings['font_family']);
        }
        if (isset($settings['custom_font'])) {
            $options['custom_font'] = sanitize_text_field($settings['custom_font']);
        }
        if (isset($settings['font_size'])) {
            $options['font_size'] = sanitize_text_field($settings['font_size']);
        }
        if (isset($settings['font_size_custom'])) {
            $options['font_size_custom'] = absint($settings['font_size_custom']);
        }

        
        if (isset($settings['table_layout'])) {
            $options['table_layout'] = sanitize_text_field($settings['table_layout']);
        }
        if (isset($settings['border_radius'])) {
            $options['border_radius'] = sanitize_text_field($settings['border_radius']);
        }

        
        if (isset($settings['custom_class'])) {
            $options['custom_class'] = sanitize_text_field($settings['custom_class']);
        }

        
        $html = \Woonutri_Label_Renderer::render_label(0, $clean_sample, $options);

        wp_send_json_success(array('html' => $html));
    }

    
    public static function render_display_tab()
    {
        ?>
        <div class="woonutri-settings-section">
            <h3><?php _e('Display Position', 'smart-nutrition-labels'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Display Location', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::display_position_field(); ?></td>
                </tr>
                <tr class="woonutri-tab-config">
                    <th scope="row"><?php _e('Tab Title', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::tab_title_field(); ?></td>
                </tr>
                <tr class="woonutri-tab-config">
                    <th scope="row"><?php _e('Tab Priority', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::tab_priority_field(); ?></td>
                </tr>
                <tr class="woonutri-tab-config">
                    <th scope="row"><?php _e('Show Tab Icon', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::tab_show_icon_field(); ?></td>
                </tr>
            </table>

            <h3><?php _e('Display Features', 'smart-nutrition-labels'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Display Format', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::display_format_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Nutri-Score Badge', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::show_nutri_score_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('UK Traffic Light Indicators', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::show_traffic_lights_field(); ?></td>
                </tr>
            </table>
        </div>
        <?php
    }

    
    public static function render_appearance_tab()
    {
        ?>
        <div class="woonutri-settings-section">
            <h3><?php _e('Theme & Colors', 'smart-nutrition-labels'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Label Theme', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::label_theme_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Primary Color', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::primary_color_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Background Color', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::background_color_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Text Color', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::text_color_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Border Color', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::border_color_field(); ?></td>
                </tr>
            </table>

            <h3><?php _e('Typography', 'smart-nutrition-labels'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Font Family', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::font_family_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Custom Google Font', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::custom_font_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Font Size', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::font_size_field(); ?></td>
                </tr>
            </table>
        </div>
        <?php
    }

    
    
    public static function render_advanced_tab()
    {
        ?>
        <div class="woonutri-settings-section">
            <h3><?php _e('Layout Options', 'smart-nutrition-labels'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Table Layout', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::table_layout_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Border Radius', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::border_radius_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Show Logo', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::show_logo_field(); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Enable Collapsible Sections', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::enable_collapsible_field(); ?></td>
                </tr>
            </table>

            <h3><?php _e('Units Configuration', 'smart-nutrition-labels'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Energy Unit', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::unit_field('energy_kcal'); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Fat Unit', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::unit_field('fat'); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Sugar Unit', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::unit_field('sugar'); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Protein Unit', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::unit_field('protein'); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Fiber Unit', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::unit_field('fiber'); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Salt Unit', 'smart-nutrition-labels'); ?></th>
                    <td><?php self::unit_field('salt'); ?></td>
                </tr>
            </table>

            <h3><?php _e('Cache Management', 'smart-nutrition-labels'); ?></h3>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Clear Cache', 'smart-nutrition-labels'); ?></th>
                    <td>
                        <button type="button" class="button woonutri-clear-cache" data-nonce="<?php echo esc_attr(wp_create_nonce('woonutri_clear_cache')); ?>">
                            <?php _e('Clear Cache', 'smart-nutrition-labels'); ?>
                        </button>
                        <span class="woonutri-cache-message" style="margin-left: 10px;"></span>
                        <p class="description"><?php _e('Clear all cached nutrition labels. Use this after making bulk changes to ensure labels are regenerated.', 'smart-nutrition-labels'); ?></p>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }

    
    
    public static function render_pro_tab()
    {
        ?>
        <div class="woonutri-settings-section">
            <h3><?php _e('Pro Features', 'smart-nutrition-labels'); ?></h3>
            <p><?php _e('Thank you for being a Pro user! Here are the premium features available to you.', 'smart-nutrition-labels'); ?></p>

            <div class="woonutri-pro-feature" style="background: #f0f7ff; border-left: 4px solid #2271b1;">
                <h4><?php _e('Premium Themes', 'smart-nutrition-labels'); ?></h4>
                <p><?php _e('Access professional nutrition label themes including FDA, EU, and UK compliance formats.', 'smart-nutrition-labels'); ?></p>
            </div>

            <div class="woonutri-pro-feature" style="background: #f0f7ff; border-left: 4px solid #2271b1;">
                <h4><?php _e('Auto %DV Calculator', 'smart-nutrition-labels'); ?></h4>
                <p><?php _e('Automatically calculate and display % Daily Values with color-coded indicators.', 'smart-nutrition-labels'); ?></p>
            </div>

            <div class="woonutri-pro-feature" style="background: #f0f7ff; border-left: 4px solid #2271b1;">
                <h4><?php _e('Smart Allergen Bolding', 'smart-nutrition-labels'); ?></h4>
                <p><?php _e('Automatically bold allergens in ingredients lists for EU/FDA compliance.', 'smart-nutrition-labels'); ?></p>
            </div>

            <div class="woonutri-pro-feature" style="background: #f0f7ff; border-left: 4px solid #2271b1;">
                <h4><?php _e('Bulk Operations', 'smart-nutrition-labels'); ?></h4>
                <p><?php _e('Edit nutrition data for multiple products at once with powerful bulk editing tools.', 'smart-nutrition-labels'); ?></p>
            </div>

            <div class="woonutri-pro-feature" style="background: #f0f7ff; border-left: 4px solid #2271b1;">
                <h4><?php _e('REST API Access', 'smart-nutrition-labels'); ?></h4>
                <p><?php _e('Integrate with external systems using comprehensive REST API endpoints.', 'smart-nutrition-labels'); ?></p>
            </div>

            <h3 style="margin-top: 30px;"><?php _e('Coming Soon', 'smart-nutrition-labels'); ?></h3>
            <p class="description"><?php _e('This AI-powered feature is in development and will be available in a future update.', 'smart-nutrition-labels'); ?></p>

            <div class="woonutri-pro-feature" style="background: #f9f9f9; border-left: 4px solid #ddd; opacity: 0.7;">
                <h4><?php _e('OCR Nutrition Scanning', 'smart-nutrition-labels'); ?> <span class="woonutri-badge" style="background: #666; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 11px; margin-left: 8px;"><?php _e('Coming Soon', 'smart-nutrition-labels'); ?></span></h4>
                <p><?php _e('Upload nutrition label images and automatically extract nutrition data using AI-powered OCR technology.', 'smart-nutrition-labels'); ?></p>
            </div>
        </div>
        <?php
    }

    
    public static function render_features_page()
    {
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        ?>
        <div class="wrap woonutri-features-wrap">
            <div class="woonutri-features-header">
                <h1><?php _e('WooNutri Features Overview', 'smart-nutrition-labels'); ?></h1>
                <p><?php _e('Discover all the powerful features available in WooNutri for creating professional nutrition labels.', 'smart-nutrition-labels'); ?>
                </p>
            </div>

            <div class="woonutri-features-grid">
                <!-- Core Features -->
                <div class="woonutri-feature-card woonutri-free-feature">
                    <div class="woonutri-feature-icon">📋</div>
                    <h3><?php _e('Basic Nutrition Labels', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Create nutrition facts labels with essential nutrients: calories, fat, carbs, sugar, protein, fiber, and salt.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div class="woonutri-feature-status woonutri-free-badge"><?php _e('FREE', 'smart-nutrition-labels'); ?></div>
                </div>

                <div class="woonutri-feature-card woonutri-free-feature">
                    <div class="woonutri-feature-icon">🎨</div>
                    <h3><?php _e('Basic Themes', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Choose from basic, minimal, and clean themes with customizable colors and typography.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div class="woonutri-feature-status woonutri-free-badge"><?php _e('FREE', 'smart-nutrition-labels'); ?></div>
                </div>

                <div class="woonutri-feature-card woonutri-free-feature">
                    <div class="woonutri-feature-icon">📍</div>
                    <h3><?php _e('Flexible Display Options', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Display nutrition labels in product summary, dedicated tab, or both locations.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div class="woonutri-feature-status woonutri-free-badge"><?php _e('FREE', 'smart-nutrition-labels'); ?></div>
                </div>

                <div class="woonutri-feature-card woonutri-free-feature">
                    <div class="woonutri-feature-icon">📱</div>
                    <h3><?php _e('Responsive Design', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Labels automatically adapt to mobile, tablet, and desktop screens with proper scaling.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div class="woonutri-feature-status woonutri-free-badge"><?php _e('FREE', 'smart-nutrition-labels'); ?></div>
                </div>

                <!-- Pro Features -->
                <div class="woonutri-feature-card <?php echo $is_pro ? 'woonutri-pro-active' : 'woonutri-pro-feature'; ?>">
                    <div class="woonutri-feature-icon">🏛️</div>
                    <h3><?php _e('FDA Compliance Labels', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Professional FDA-compliant nutrition labels that match real product packaging standards.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div
                        class="woonutri-feature-status <?php echo $is_pro ? 'woonutri-pro-active-badge' : 'woonutri-pro-badge'; ?>">
                        <?php echo $is_pro ? __('ACTIVE', 'smart-nutrition-labels') : __('PRO', 'smart-nutrition-labels'); ?>
                    </div>
                </div>

                <div class="woonutri-feature-card <?php echo $is_pro ? 'woonutri-pro-active' : 'woonutri-pro-feature'; ?>">
                    <div class="woonutri-feature-icon">🇪🇺</div>
                    <h3><?php _e('EU & UK Standards', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('European and UK nutrition information formats with proper Reference Intake calculations.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div
                        class="woonutri-feature-status <?php echo $is_pro ? 'woonutri-pro-active-badge' : 'woonutri-pro-badge'; ?>">
                        <?php echo $is_pro ? __('ACTIVE', 'smart-nutrition-labels') : __('PRO', 'smart-nutrition-labels'); ?>
                    </div>
                </div>

                <div class="woonutri-feature-card <?php echo $is_pro ? 'woonutri-pro-active' : 'woonutri-pro-feature'; ?>">
                    <div class="woonutri-feature-icon">🔍</div>
                    <h3><?php _e('AI OCR Scanning', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Upload nutrition label images and automatically extract data using AI-powered OCR technology.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div
                        class="woonutri-feature-status <?php echo $is_pro ? 'woonutri-pro-active-badge' : 'woonutri-pro-badge'; ?>">
                        <?php echo $is_pro ? __('ACTIVE', 'smart-nutrition-labels') : __('PRO', 'smart-nutrition-labels'); ?>
                    </div>
                </div>

                <div class="woonutri-feature-card <?php echo $is_pro ? 'woonutri-pro-active' : 'woonutri-pro-feature'; ?>">
                    <div class="woonutri-feature-icon">📊</div>
                    <h3><?php _e('Barcode Database Lookup', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Instantly lookup nutrition information by scanning product barcodes from OpenFoodFacts database.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div
                        class="woonutri-feature-status <?php echo $is_pro ? 'woonutri-pro-active-badge' : 'woonutri-pro-badge'; ?>">
                        <?php echo $is_pro ? __('ACTIVE', 'smart-nutrition-labels') : __('PRO', 'smart-nutrition-labels'); ?>
                    </div>
                </div>

                <div class="woonutri-feature-card <?php echo $is_pro ? 'woonutri-pro-active' : 'woonutri-pro-feature'; ?>">
                    <div class="woonutri-feature-icon">🎨</div>
                    <h3><?php _e('Premium Themes', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Access to classic, modern, premium, and dark themes with advanced styling options.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div
                        class="woonutri-feature-status <?php echo $is_pro ? 'woonutri-pro-active-badge' : 'woonutri-pro-badge'; ?>">
                        <?php echo $is_pro ? __('ACTIVE', 'smart-nutrition-labels') : __('PRO', 'smart-nutrition-labels'); ?>
                    </div>
                </div>

                <div class="woonutri-feature-card <?php echo $is_pro ? 'woonutri-pro-active' : 'woonutri-pro-feature'; ?>">
                    <div class="woonutri-feature-icon">📈</div>
                    <h3><?php _e('Bulk Operations', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Edit nutrition data for multiple products at once with powerful bulk editing tools.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div
                        class="woonutri-feature-status <?php echo $is_pro ? 'woonutri-pro-active-badge' : 'woonutri-pro-badge'; ?>">
                        <?php echo $is_pro ? __('ACTIVE', 'smart-nutrition-labels') : __('PRO', 'smart-nutrition-labels'); ?>
                    </div>
                </div>

                <div class="woonutri-feature-card <?php echo $is_pro ? 'woonutri-pro-active' : 'woonutri-pro-feature'; ?>">
                    <div class="woonutri-feature-icon">🔧</div>
                    <h3><?php _e('REST API Access', 'smart-nutrition-labels'); ?></h3>
                    <p><?php _e('Integrate with external systems using comprehensive REST API endpoints for nutrition data.', 'smart-nutrition-labels'); ?>
                    </p>
                    <div
                        class="woonutri-feature-status <?php echo $is_pro ? 'woonutri-pro-active-badge' : 'woonutri-pro-badge'; ?>">
                        <?php echo $is_pro ? __('ACTIVE', 'smart-nutrition-labels') : __('PRO', 'smart-nutrition-labels'); ?>
                    </div>
                </div>
            </div>

            <?php if (!$is_pro): ?>
                <div class="woonutri-upgrade-cta">
                    <h2><?php _e('Ready to unlock all features?', 'smart-nutrition-labels'); ?></h2>
                    <p><?php _e('Upgrade to WooNutri Pro to access all advanced features and create professional nutrition labels that comply with international standards.', 'smart-nutrition-labels'); ?>
                    </p>
                    <a href="<?php echo woonutri_fs()->get_upgrade_url(); ?>" class="button button-primary button-hero">
                        <?php _e('Upgrade to Pro Now', 'smart-nutrition-labels'); ?>
                    </a>
                </div>
            <?php endif; ?>
        </div>

        <style>
            .woonutri-features-wrap {
                max-width: 1200px;
                margin: 20px auto;
            }

            .woonutri-features-header {
                background: linear-gradient(135deg, #1e3a5f 0%, #2a5298 100%);
                color: #fff;
                padding: 40px;
                border-radius: 8px;
                margin-bottom: 30px;
                text-align: center;
            }

            .woonutri-features-header h1 {
                margin: 0 0 15px 0;
                font-size: 32px;
                font-weight: 400;
            }

            .woonutri-features-header p {
                margin: 0;
                font-size: 16px;
                opacity: 0.9;
            }

            .woonutri-features-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
                gap: 20px;
                margin-bottom: 40px;
            }

            .woonutri-feature-card {
                background: #fff;
                border: 1px solid #c3c4c7;
                border-radius: 8px;
                padding: 30px;
                position: relative;
                transition: all 0.3s ease;
                box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
            }

            .woonutri-feature-card:hover {
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            }

            .woonutri-feature-icon {
                font-size: 48px;
                margin-bottom: 20px;
            }

            .woonutri-feature-card h3 {
                margin: 0 0 15px 0;
                font-size: 20px;
                font-weight: 600;
                color: #1d2327;
            }

            .woonutri-feature-card p {
                margin: 0 0 20px 0;
                color: #646970;
                line-height: 1.6;
            }

            .woonutri-feature-status {
                position: absolute;
                top: 15px;
                right: 15px;
                padding: 4px 12px;
                border-radius: 4px;
                font-size: 11px;
                font-weight: 600;
                text-transform: uppercase;
            }

            .woonutri-free-badge {
                background: #00a32a;
                color: #fff;
            }

            .woonutri-pro-badge {
                background: #ff6b35;
                color: #fff;
            }

            .woonutri-pro-active-badge {
                background: #1e40af;
                color: #fff;
            }

            .woonutri-pro-feature {
                opacity: 0.7;
            }

            .woonutri-pro-active {
                border-color: #1e40af;
                background: linear-gradient(135deg, #f8faff 0%, #e0f2fe 100%);
            }

            .woonutri-upgrade-cta {
                background: linear-gradient(135deg, #ff6b35 0%, #f7931e 100%);
                color: #fff;
                padding: 40px;
                border-radius: 8px;
                text-align: center;
            }

            .woonutri-upgrade-cta h2 {
                margin: 0 0 15px 0;
                color: #fff;
            }

            .woonutri-upgrade-cta p {
                margin: 0 0 25px 0;
                opacity: 0.9;
            }

            .woonutri-upgrade-cta .button-hero {
                font-size: 16px;
                padding: 12px 30px;
                background: #fff;
                color: #ff6b35;
                border: none;
                font-weight: 600;
            }

            .woonutri-upgrade-cta .button-hero:hover,
            .woonutri-upgrade-cta .button-hero:focus {
                background: #f0f0f0;
                color: #ff6b35;
            }
        </style>
        <?php
    }

    
    public static function ajax_dismiss_notice()
    {
        
        if (!wp_verify_nonce($_POST['nonce'], 'woonutri_dismiss_notice')) {
            wp_die('Security check failed');
        }

        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }

        $period = sanitize_text_field($_POST['period'] ?? 'week');
        $user_id = get_current_user_id();

        
        switch ($period) {
            case 'week':
                $dismiss_until = time() + (7 * 24 * 60 * 60); 
                break;
            case 'month':
                $dismiss_until = time() + (30 * 24 * 60 * 60); 
                break;
            case 'forever':
                $dismiss_until = time() + (10 * 365 * 24 * 60 * 60); 
                break;
            default:
                $dismiss_until = time() + (7 * 24 * 60 * 60); 
        }

        
        update_user_meta($user_id, 'woonutri_upgrade_notice_dismissed_until', $dismiss_until);

        
        delete_user_meta($user_id, 'woonutri_upgrade_notice_dismissed');

        wp_send_json_success(['message' => __('Notice dismissed', 'smart-nutrition-labels')]);
    }

    public static function ajax_clear_cache()
    {
        check_ajax_referer('woonutri_clear_cache', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Unauthorized', 'smart-nutrition-labels')]);
        }

        if (class_exists('Woonutri_Cache')) {
            Woonutri_Cache::flush();
            wp_send_json_success(['message' => __('Cache cleared successfully!', 'smart-nutrition-labels')]);
        } else {
            wp_send_json_error(['message' => __('Cache class not found.', 'smart-nutrition-labels')]);
        }
    }
}