<?php
/**
 * @package WooNutri
 * @since 1.2.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Woonutri_Allergens
{

    public static function get_eu_allergens()
    {
        return [
            'gluten' => [
                'name' => __('Gluten', 'smart-nutrition-labels'),
                'description' => __('Cereals containing gluten (wheat, rye, barley, oats, spelt, kamut)', 'smart-nutrition-labels'),
                'keywords' => ['wheat', 'rye', 'barley', 'oats', 'spelt', 'kamut', 'flour', 'bread', 'pasta', 'couscous', 'bulgur', 'semolina'],
                'icon' => '🌾',
            ],
            'crustaceans' => [
                'name' => __('Crustaceans', 'smart-nutrition-labels'),
                'description' => __('Crustaceans and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['crab', 'lobster', 'shrimp', 'prawn', 'crayfish', 'langoustine', 'scampi'],
                'icon' => '🦐',
            ],
            'eggs' => [
                'name' => __('Eggs', 'smart-nutrition-labels'),
                'description' => __('Eggs and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['egg', 'albumin', 'globulin', 'lysozyme', 'mayonnaise', 'meringue', 'ovalbumin', 'ovomucin'],
                'icon' => '🥚',
            ],
            'fish' => [
                'name' => __('Fish', 'smart-nutrition-labels'),
                'description' => __('Fish and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['fish', 'cod', 'salmon', 'tuna', 'anchovy', 'sardine', 'fish sauce', 'fish oil', 'omega-3'],
                'icon' => '🐟',
            ],
            'peanuts' => [
                'name' => __('Peanuts', 'smart-nutrition-labels'),
                'description' => __('Peanuts and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['peanut', 'groundnut', 'arachis', 'peanut butter', 'peanut oil'],
                'icon' => '🥜',
            ],
            'soybeans' => [
                'name' => __('Soybeans', 'smart-nutrition-labels'),
                'description' => __('Soybeans and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['soy', 'soya', 'tofu', 'tempeh', 'edamame', 'miso', 'soy sauce', 'soy lecithin', 'soybean oil'],
                'icon' => '🫘',
            ],
            'milk' => [
                'name' => __('Milk', 'smart-nutrition-labels'),
                'description' => __('Milk and products thereof (including lactose)', 'smart-nutrition-labels'),
                'keywords' => ['milk', 'lactose', 'casein', 'whey', 'cream', 'butter', 'cheese', 'yogurt', 'ghee', 'lactalbumin'],
                'icon' => '🥛',
            ],
            'nuts' => [
                'name' => __('Tree Nuts', 'smart-nutrition-labels'),
                'description' => __('Tree nuts (almonds, hazelnuts, walnuts, cashews, pecans, brazil nuts, pistachios, macadamia)', 'smart-nutrition-labels'),
                'keywords' => ['almond', 'hazelnut', 'walnut', 'cashew', 'pecan', 'brazil nut', 'pistachio', 'macadamia', 'chestnut', 'nut'],
                'icon' => '🌰',
            ],
            'celery' => [
                'name' => __('Celery', 'smart-nutrition-labels'),
                'description' => __('Celery and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['celery', 'celeriac', 'celery salt', 'celery seed'],
                'icon' => '🥬',
            ],
            'mustard' => [
                'name' => __('Mustard', 'smart-nutrition-labels'),
                'description' => __('Mustard and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['mustard', 'mustard seed', 'mustard oil', 'mustard powder'],
                'icon' => '🟡',
            ],
            'sesame' => [
                'name' => __('Sesame', 'smart-nutrition-labels'),
                'description' => __('Sesame seeds and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['sesame', 'tahini', 'sesame oil', 'sesame seed', 'halvah'],
                'icon' => '⚪',
            ],
            'sulphites' => [
                'name' => __('Sulphites', 'smart-nutrition-labels'),
                'description' => __('Sulphur dioxide and sulphites (>10mg/kg or >10mg/L)', 'smart-nutrition-labels'),
                'keywords' => ['sulphite', 'sulfite', 'sulphur dioxide', 'sulfur dioxide', 'E220', 'E221', 'E222', 'E223', 'E224', 'E225', 'E226', 'E227', 'E228'],
                'icon' => '⚗️',
            ],
            'lupin' => [
                'name' => __('Lupin', 'smart-nutrition-labels'),
                'description' => __('Lupin and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['lupin', 'lupine', 'lupin flour', 'lupin seed'],
                'icon' => '🌸',
            ],
            'molluscs' => [
                'name' => __('Molluscs', 'smart-nutrition-labels'),
                'description' => __('Molluscs and products thereof', 'smart-nutrition-labels'),
                'keywords' => ['mollusc', 'mollusk', 'squid', 'octopus', 'snail', 'mussel', 'oyster', 'clam', 'scallop'],
                'icon' => '🦑',
            ],
        ];
    }

    public static function get_dietary_preferences()
    {
        return [
            'vegan' => [
                'name' => __('Vegan', 'smart-nutrition-labels'),
                'description' => __('Contains no animal products or by-products', 'smart-nutrition-labels'),
                'icon' => '🌱',
                'type' => 'positive',
            ],
            'vegetarian' => [
                'name' => __('Vegetarian', 'smart-nutrition-labels'),
                'description' => __('Contains no meat or fish, may contain dairy/eggs', 'smart-nutrition-labels'),
                'icon' => '🥬',
                'type' => 'positive',
            ],
            'gluten_free' => [
                'name' => __('Gluten-Free', 'smart-nutrition-labels'),
                'description' => __('Certified gluten-free product', 'smart-nutrition-labels'),
                'icon' => '🌾',
                'type' => 'positive',
            ],
            'dairy_free' => [
                'name' => __('Dairy-Free', 'smart-nutrition-labels'),
                'description' => __('Contains no milk or dairy products', 'smart-nutrition-labels'),
                'icon' => '🥛',
                'type' => 'positive',
            ],
            'halal' => [
                'name' => __('Halal', 'smart-nutrition-labels'),
                'description' => __('Permissible under Islamic dietary law', 'smart-nutrition-labels'),
                'icon' => '☪️',
                'type' => 'positive',
            ],
            'kosher' => [
                'name' => __('Kosher', 'smart-nutrition-labels'),
                'description' => __('Permissible under Jewish dietary law', 'smart-nutrition-labels'),
                'icon' => '✡️',
                'type' => 'positive',
            ],
            'pork_free' => [
                'name' => __('Pork-Free', 'smart-nutrition-labels'),
                'description' => __('Contains no pork or pork-derived ingredients', 'smart-nutrition-labels'),
                'icon' => '🐷',
                'type' => 'positive',
            ],
            'organic' => [
                'name' => __('Organic', 'smart-nutrition-labels'),
                'description' => __('Certified organic product', 'smart-nutrition-labels'),
                'icon' => '🌿',
                'type' => 'positive',
            ],
            'non_gmo' => [
                'name' => __('Non-GMO', 'smart-nutrition-labels'),
                'description' => __('No genetically modified ingredients', 'smart-nutrition-labels'),
                'icon' => '🧬',
                'type' => 'positive',
            ],
            'keto' => [
                'name' => __('Keto-Friendly', 'smart-nutrition-labels'),
                'description' => __('Suitable for ketogenic diet (low carb)', 'smart-nutrition-labels'),
                'icon' => '🥑',
                'type' => 'positive',
            ],
            'low_sugar' => [
                'name' => __('Low Sugar', 'smart-nutrition-labels'),
                'description' => __('Low sugar content', 'smart-nutrition-labels'),
                'icon' => '🍬',
                'type' => 'positive',
            ],
            'zero_sugar' => [
                'name' => __('Zero Sugar', 'smart-nutrition-labels'),
                'description' => __('Contains no sugar or added sugars', 'smart-nutrition-labels'),
                'icon' => '🚫',
                'type' => 'positive',
            ],
            'high_protein' => [
                'name' => __('High Protein', 'smart-nutrition-labels'),
                'description' => __('High protein content', 'smart-nutrition-labels'),
                'icon' => '💪',
                'type' => 'positive',
            ],
            'contains_pork' => [
                'name' => __('Contains Pork', 'smart-nutrition-labels'),
                'description' => __('Contains pork or pork-derived ingredients', 'smart-nutrition-labels'),
                'icon' => '🐖',
                'type' => 'warning',
            ],
            'contains_alcohol' => [
                'name' => __('Contains Alcohol', 'smart-nutrition-labels'),
                'description' => __('Contains alcohol or alcohol-derived ingredients', 'smart-nutrition-labels'),
                'icon' => '🍷',
                'type' => 'warning',
            ],
            'raw' => [
                'name' => __('Raw', 'smart-nutrition-labels'),
                'description' => __('Raw or unprocessed food product', 'smart-nutrition-labels'),
                'icon' => '🥗',
                'type' => 'positive',
            ],
            'whole_grain' => [
                'name' => __('Whole Grain', 'smart-nutrition-labels'),
                'description' => __('Made with whole grain ingredients', 'smart-nutrition-labels'),
                'icon' => '🌾',
                'type' => 'positive',
            ],
        ];
    }

    public static function get_us_allergens()
    {
        return [
            'milk' => [
                'name' => __('Milk', 'smart-nutrition-labels'),
                'keywords' => ['milk', 'butter', 'cheese', 'cream', 'yogurt', 'casein', 'whey', 'lactose', 'lactalbumin'],
            ],
            'eggs' => [
                'name' => __('Eggs', 'smart-nutrition-labels'),
                'keywords' => ['egg', 'albumin', 'globulin', 'mayonnaise', 'meringue'],
            ],
            'fish' => [
                'name' => __('Fish', 'smart-nutrition-labels'),
                'keywords' => ['fish', 'bass', 'cod', 'flounder', 'haddock', 'halibut', 'salmon', 'tilapia', 'trout', 'tuna'],
            ],
            'crustacean_shellfish' => [
                'name' => __('Crustacean Shellfish', 'smart-nutrition-labels'),
                'keywords' => ['crab', 'crayfish', 'lobster', 'prawn', 'shrimp'],
            ],
            'tree_nuts' => [
                'name' => __('Tree Nuts', 'smart-nutrition-labels'),
                'keywords' => ['almond', 'cashew', 'hazelnut', 'macadamia', 'pecan', 'pine nut', 'pistachio', 'walnut'],
            ],
            'peanuts' => [
                'name' => __('Peanuts', 'smart-nutrition-labels'),
                'keywords' => ['peanut', 'groundnut', 'peanut butter', 'peanut oil'],
            ],
            'wheat' => [
                'name' => __('Wheat', 'smart-nutrition-labels'),
                'keywords' => ['wheat', 'flour', 'bread', 'pasta', 'semolina', 'spelt', 'durum'],
            ],
            'soybeans' => [
                'name' => __('Soybeans', 'smart-nutrition-labels'),
                'keywords' => ['soy', 'soybean', 'tofu', 'tempeh', 'edamame', 'miso', 'soy sauce'],
            ],
            'sesame' => [
                'name' => __('Sesame', 'smart-nutrition-labels'),
                'keywords' => ['sesame', 'tahini', 'sesame oil', 'sesame seed'],
            ],
        ];
    }

    public static function get_allergens($region = 'eu')
    {
        switch ($region) {
            case 'us':
            case 'usa':
                return self::get_us_allergens();
            case 'eu':
            case 'uk':
            default:
                return self::get_eu_allergens();
        }
    }

    public static function get_autocomplete_list($region = 'eu')
    {
        $allergens = self::get_allergens($region);
        $list = [];

        foreach ($allergens as $key => $allergen) {
            $list[] = [
                'value' => $allergen['name'],
                'label' => $allergen['name'],
                'key' => $key,
                'icon' => $allergen['icon'] ?? '',
            ];
        }

        return $list;
    }

    public static function get_all_keywords($region = 'eu')
    {
        $allergens = self::get_allergens($region);
        $keywords = [];

        foreach ($allergens as $key => $allergen) {
            foreach ($allergen['keywords'] as $keyword) {
                $keywords[strtolower($keyword)] = $key;
            }
        }

        return $keywords;
    }

    public static function detect_allergens($ingredients_text, $region = 'eu')
    {
        if (empty($ingredients_text)) {
            return [];
        }

        $keywords = self::get_all_keywords($region);
        $allergens = self::get_allergens($region);
        $text_lower = strtolower($ingredients_text);
        $detected = [];

        foreach ($keywords as $keyword => $allergen_key) {
            if (strpos($text_lower, $keyword) !== false) {
                if (!isset($detected[$allergen_key])) {
                    $detected[$allergen_key] = [
                        'name' => $allergens[$allergen_key]['name'],
                        'icon' => $allergens[$allergen_key]['icon'] ?? '',
                        'matched_keywords' => [],
                    ];
                }
                $detected[$allergen_key]['matched_keywords'][] = $keyword;
            }
        }

        return $detected;
    }

    public static function format_allergens($allergen_keys, $region = 'eu', $format = 'text')
    {
        $allergens = self::get_allergens($region);
        $formatted = [];

        foreach ((array) $allergen_keys as $key) {
            if (isset($allergens[$key])) {
                switch ($format) {
                    case 'icon':
                        $formatted[] = ($allergens[$key]['icon'] ?? '') . ' ' . $allergens[$key]['name'];
                        break;
                    case 'html':
                        $icon = $allergens[$key]['icon'] ?? '';
                        $formatted[] = '<span class="woonutri-allergen" data-allergen="' . esc_attr($key) . '">' .
                            $icon . ' <strong>' . esc_html($allergens[$key]['name']) . '</strong></span>';
                        break;
                    case 'text':
                    default:
                        $formatted[] = $allergens[$key]['name'];
                        break;
                }
            } else {
                $formatted[] = ucfirst($key);
            }
        }

        return $format === 'html' ? implode(', ', $formatted) : implode(', ', $formatted);
    }

    public static function highlight_allergens_in_text($ingredients_text, $region = 'eu')
    {
        if (empty($ingredients_text)) {
            return $ingredients_text;
        }

        $keywords = self::get_all_keywords($region);
        $highlighted = $ingredients_text;

        $sorted_keywords = array_keys($keywords);
        usort($sorted_keywords, function ($a, $b) {
            return strlen($b) - strlen($a);
        });

        foreach ($sorted_keywords as $keyword) {
            $pattern = '/\b(' . preg_quote($keyword, '/') . ')\b/i';
            $highlighted = preg_replace($pattern, '<strong class="woonutri-allergen-highlight">$1</strong>', $highlighted);
        }

        return $highlighted;
    }

    public static function get_may_contain_phrases()
    {
        return [
            __('May contain traces of', 'smart-nutrition-labels'),
            __('Produced in a facility that also handles', 'smart-nutrition-labels'),
            __('May contain', 'smart-nutrition-labels'),
            __('Manufactured on equipment shared with', 'smart-nutrition-labels'),
            __('Cross-contamination possible with', 'smart-nutrition-labels'),
        ];
    }
}
