<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Woonutri_Analytics {

    const TABLE_NAME = 'woonutri_analytics';

    public static function init() {
        if ( ! self::is_pro() ) {
            return;
        }

        add_action( 'wp_ajax_woonutri_track_view', [ __CLASS__, 'ajax_track_view' ] );
        add_action( 'wp_ajax_nopriv_woonutri_track_view', [ __CLASS__, 'ajax_track_view' ] );
        add_action( 'wp_enqueue_scripts', [ __CLASS__, 'enqueue_tracking_script' ] );
    }

    private static function is_pro() {
        return function_exists( 'woonutri_fs' ) && woonutri_fs()->can_use_premium_code();
    }

    public static function create_table() {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            view_date date NOT NULL,
            view_count int(11) unsigned NOT NULL DEFAULT 1,
            unique_views int(11) unsigned NOT NULL DEFAULT 1,
            PRIMARY KEY (id),
            UNIQUE KEY product_date (product_id, view_date),
            KEY product_id (product_id),
            KEY view_date (view_date)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );
    }

    public static function enqueue_tracking_script() {
        if ( ! is_product() ) {
            return;
        }

        wp_add_inline_script( 'woonutri-nutrition-label', self::get_tracking_script() );
    }

    private static function get_tracking_script() {
        $nonce = wp_create_nonce( 'woonutri_track_view' );
        $ajax_url = admin_url( 'admin-ajax.php' );

        return "
        (function() {
            var tracked = false;
            var label = document.querySelector('.woonutri-label');
            if (!label) return;

            var productId = label.dataset.productId;
            if (!productId) return;

            function trackView() {
                if (tracked) return;
                tracked = true;

                var formData = new FormData();
                formData.append('action', 'woonutri_track_view');
                formData.append('nonce', '{$nonce}');
                formData.append('product_id', productId);

                fetch('{$ajax_url}', {
                    method: 'POST',
                    body: formData,
                    credentials: 'same-origin'
                }).catch(function() {});
            }

            if ('IntersectionObserver' in window) {
                var observer = new IntersectionObserver(function(entries) {
                    entries.forEach(function(entry) {
                        if (entry.isIntersecting) {
                            trackView();
                            observer.disconnect();
                        }
                    });
                }, { threshold: 0.5 });
                observer.observe(label);
            } else {
                trackView();
            }
        })();
        ";
    }

    public static function ajax_track_view() {
        check_ajax_referer( 'woonutri_track_view', 'nonce' );

        $product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;

        if ( ! $product_id ) {
            wp_send_json_error( 'Invalid product ID' );
        }

        self::record_view( $product_id );

        wp_send_json_success();
    }

    public static function record_view( $product_id ) {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;
        $today = current_time( 'Y-m-d' );

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table_name}'" ) !== $table_name ) {
            self::create_table();
        }

        $updated = $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$table_name}
                SET view_count = view_count + 1
                WHERE product_id = %d AND view_date = %s",
                $product_id,
                $today
            )
        );

        if ( 0 === $updated ) {
            $wpdb->insert(
                $table_name,
                [
                    'product_id' => $product_id,
                    'view_date' => $today,
                    'view_count' => 1,
                    'unique_views' => 1,
                ],
                [ '%d', '%s', '%d', '%d' ]
            );
        }
    }

    public static function get_product_views( $product_id, $days = 0 ) {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;

        if ( $days > 0 ) {
            $date_limit = date( 'Y-m-d', strtotime( "-{$days} days" ) );
            return (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT SUM(view_count) FROM {$table_name}
                    WHERE product_id = %d AND view_date >= %s",
                    $product_id,
                    $date_limit
                )
            );
        }

        return (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT SUM(view_count) FROM {$table_name} WHERE product_id = %d",
                $product_id
            )
        );
    }

    public static function get_top_products( $limit = 10, $days = 30 ) {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;

        $where_clause = '';
        if ( $days > 0 ) {
            $date_limit = date( 'Y-m-d', strtotime( "-{$days} days" ) );
            $where_clause = $wpdb->prepare( " WHERE view_date >= %s", $date_limit );
        }

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT product_id, SUM(view_count) as total_views
                FROM {$table_name}
                {$where_clause}
                GROUP BY product_id
                ORDER BY total_views DESC
                LIMIT %d",
                $limit
            )
        );

        $products = [];
        foreach ( $results as $row ) {
            $product = wc_get_product( $row->product_id );
            if ( $product ) {
                $products[] = [
                    'id' => $row->product_id,
                    'name' => $product->get_name(),
                    'views' => (int) $row->total_views,
                    'edit_url' => get_edit_post_link( $row->product_id, 'raw' ),
                    'view_url' => get_permalink( $row->product_id ),
                ];
            }
        }

        return $products;
    }

    public static function get_total_views( $days = 0 ) {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;

        if ( $days > 0 ) {
            $date_limit = date( 'Y-m-d', strtotime( "-{$days} days" ) );
            return (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT SUM(view_count) FROM {$table_name} WHERE view_date >= %s",
                    $date_limit
                )
            );
        }

        return (int) $wpdb->get_var( "SELECT SUM(view_count) FROM {$table_name}" );
    }

    public static function get_daily_views( $days = 30 ) {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;
        $date_limit = date( 'Y-m-d', strtotime( "-{$days} days" ) );

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT view_date, SUM(view_count) as total_views
                FROM {$table_name}
                WHERE view_date >= %s
                GROUP BY view_date
                ORDER BY view_date ASC",
                $date_limit
            )
        );

        $daily_views = [];
        $current_date = new DateTime( $date_limit );
        $end_date = new DateTime( current_time( 'Y-m-d' ) );

        while ( $current_date <= $end_date ) {
            $date_str = $current_date->format( 'Y-m-d' );
            $daily_views[ $date_str ] = 0;
            $current_date->modify( '+1 day' );
        }

        foreach ( $results as $row ) {
            $daily_views[ $row->view_date ] = (int) $row->total_views;
        }

        return $daily_views;
    }

    public static function get_dashboard_summary() {
        return [
            'total_views_7d' => self::get_total_views( 7 ),
            'total_views_30d' => self::get_total_views( 30 ),
            'total_views_all' => self::get_total_views( 0 ),
            'top_products' => self::get_top_products( 5, 30 ),
            'daily_views' => self::get_daily_views( 30 ),
        ];
    }

    public static function cleanup_old_data( $days_to_keep = 365 ) {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;
        $cutoff_date = date( 'Y-m-d', strtotime( "-{$days_to_keep} days" ) );

        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$table_name} WHERE view_date < %s",
                $cutoff_date
            )
        );
    }
}
