<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Woonutri_Cache {

    const CACHE_GROUP = 'woonutri';

    const DEFAULT_EXPIRATION = HOUR_IN_SECONDS;

    public static function is_pro_cache_enabled() {
        if ( ! function_exists( 'woonutri_fs' ) ) {
            return false;
        }
        return woonutri_fs()->can_use_premium_code();
    }

    public static function has_object_cache() {
        if ( ! self::is_pro_cache_enabled() ) {
            return false;
        }
        return wp_using_ext_object_cache();
    }

    public static function get( $key, $default = false ) {
        $full_key = self::get_full_key( $key );

        if ( self::has_object_cache() ) {
            $found = false;
            $value = wp_cache_get( $full_key, self::CACHE_GROUP, false, $found );
            return $found ? $value : $default;
        }

        $value = get_transient( $full_key );
        return false !== $value ? $value : $default;
    }

    public static function set( $key, $value, $expiration = null ) {
        $full_key = self::get_full_key( $key );
        $expiration = $expiration ?? self::DEFAULT_EXPIRATION;

        if ( self::has_object_cache() ) {
            return wp_cache_set( $full_key, $value, self::CACHE_GROUP, $expiration );
        }

        return set_transient( $full_key, $value, $expiration );
    }

    public static function delete( $key ) {
        $full_key = self::get_full_key( $key );

        if ( self::has_object_cache() ) {
            return wp_cache_delete( $full_key, self::CACHE_GROUP );
        }

        return delete_transient( $full_key );
    }

    public static function flush() {
        if ( self::has_object_cache() ) {
            if ( function_exists( 'wp_cache_flush_group' ) ) {
                return wp_cache_flush_group( self::CACHE_GROUP );
            }
            self::delete_known_keys();
            return true;
        }

        return self::delete_all_transients();
    }

    private static function delete_all_transients() {
        global $wpdb;

        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like( '_transient_woonutri_' ) . '%'
            )
        );

        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like( '_transient_timeout_woonutri_' ) . '%'
            )
        );

        return true;
    }

    private static function delete_known_keys() {
        $known_keys = [
            'dashboard_stats',
            'filtered_ids',
            'compliance_stats',
            'nutri_score_dist',
        ];

        foreach ( $known_keys as $key ) {
            wp_cache_delete( self::get_full_key( $key ), self::CACHE_GROUP );
        }
    }

    private static function get_full_key( $key ) {
        return 'woonutri_' . $key;
    }

    public static function remember( $key, $callback, $expiration = null ) {
        $value = self::get( $key );

        if ( false !== $value ) {
            return $value;
        }

        $value = $callback();
        self::set( $key, $value, $expiration );

        return $value;
    }

    public static function invalidate_dashboard() {
        $options = get_option( 'woonutri_options', [] );
        $filter = [
            'mode' => isset( $options['category_filter_mode'] ) ? $options['category_filter_mode'] : 'all',
            'categories' => isset( $options['category_filter_ids'] ) ? array_map( 'intval', (array) $options['category_filter_ids'] ) : [],
        ];

        $filter_hash = md5( json_encode( $filter ) );

        self::delete( 'dashboard_stats_' . $filter_hash );
        self::delete( 'filtered_ids_' . $filter_hash );

        if ( ! self::has_object_cache() ) {
            global $wpdb;
            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
                    $wpdb->esc_like( '_transient_woonutri_dashboard_stats_' ) . '%',
                    $wpdb->esc_like( '_transient_woonutri_filtered_ids_' ) . '%'
                )
            );
            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
                    $wpdb->esc_like( '_transient_timeout_woonutri_dashboard_stats_' ) . '%',
                    $wpdb->esc_like( '_transient_timeout_woonutri_filtered_ids_' ) . '%'
                )
            );
        }
    }

    public static function get_cache_info() {
        $object_cache_available = wp_using_ext_object_cache();
        $is_pro = self::is_pro_cache_enabled();

        return [
            'type' => self::has_object_cache() ? 'object_cache' : 'transients',
            'object_cache_available' => $object_cache_available,
            'object_cache_enabled' => self::has_object_cache(),
            'is_pro' => $is_pro,
            'group' => self::CACHE_GROUP,
            'default_expiration' => self::DEFAULT_EXPIRATION,
            'status_message' => self::get_cache_status_message(),
        ];
    }

    public static function get_cache_status_message() {
        $object_cache_available = wp_using_ext_object_cache();
        $is_pro = self::is_pro_cache_enabled();

        if ( $is_pro && $object_cache_available ) {
            return __( 'Using Redis/Memcached object cache (Pro)', 'woonutri' );
        } elseif ( $is_pro && ! $object_cache_available ) {
            return __( 'Using database transients (Pro - object cache not detected)', 'woonutri' );
        } elseif ( ! $is_pro && $object_cache_available ) {
            return __( 'Using database transients (Upgrade to Pro for Redis/Memcached support)', 'woonutri' );
        } else {
            return __( 'Using database transients', 'woonutri' );
        }
    }
}
