<?php
class Woonutri_Meta_Handler
{
    public static $fields = [
        'energy_kcal',
        'energy_kj',

        'fat',
        'saturated_fat',
        'monounsaturated_fat',
        'polyunsaturated_fat',
        'trans_fat',
        'cholesterol',
        'sodium',

        'carbs',
        'fiber',
        'sugar',
        'added_sugars',
        'sugar_alcohols',
        'starch',

        'protein',
        'salt',
        'potassium',

        'vitamin_d',
        'calcium',
        'iron',

        'vitamin_a',
        'vitamin_e',
        'vitamin_k',

        'vitamin_c',
        'thiamin',
        'riboflavin',
        'niacin',
        'vitamin_b1',
        'vitamin_b2',
        'vitamin_b3',
        'pantothenic_acid',
        'vitamin_b6',
        'biotin',
        'folate',
        'vitamin_b12',
        'choline',

        'phosphorus',
        'magnesium',
        'chloride',

        'iodine',
        'zinc',
        'selenium',
        'copper',
        'manganese',
        'chromium',
        'molybdenum',
        'fluoride',

        'omega_3',
        'omega_6',
        'dha',
        'epa',
        'caffeine',
        'alcohol',
        'organic_acids',
        'beta_carotene',
        'lycopene',
        'lutein',
        'zeaxanthin',

        'histidine',
        'isoleucine',
        'leucine',
        'lysine',
        'methionine',
        'phenylalanine',
        'threonine',
        'tryptophan',
        'valine',
        'alanine',
        'arginine',
        'asparagine',
        'aspartic_acid',
        'cysteine',
        'glutamic_acid',
        'glutamine',
        'glycine',
        'proline',
        'serine',
        'tyrosine',

        'serving_size',
        'servings_per_container',
        'net_weight',
        'serving_per_package',

        'nutri_score',
        'nutri_score_points',
        'health_star_rating',
        'traffic_light_overall',

        'vitamins',
        'ingredients',
        'allergens',
        'may_contain',
        'dietary_preferences',
        'barcode',
        'preparation_instructions',
        'storage_instructions',
        'country_of_origin'
    ];

    public static function get_field_hierarchy()
    {
        return [
            'energy_kcal' => ['label' => 'Energy (kcal)', 'type' => 'energy', 'category' => 'basic', 'required' => true],
            'energy_kj' => ['label' => 'Energy (kJ)', 'type' => 'energy', 'category' => 'basic', 'required' => true],

            'fat' => [
                'label' => 'Total Fat',
                'type' => 'macronutrient',
                'category' => 'basic',
                'required' => true,
                'children' => [
                    'saturated_fat' => ['label' => 'Saturated Fat', 'type' => 'nested', 'category' => 'basic'],
                    'monounsaturated_fat' => ['label' => 'Monounsaturated Fat', 'type' => 'nested', 'category' => 'detailed'],
                    'polyunsaturated_fat' => ['label' => 'Polyunsaturated Fat', 'type' => 'nested', 'category' => 'detailed'],
                    'trans_fat' => ['label' => 'Trans Fat', 'type' => 'nested', 'category' => 'basic'],
                    'omega_3' => ['label' => 'Omega-3 Fatty Acids', 'type' => 'nested', 'category' => 'specialty'],
                    'omega_6' => ['label' => 'Omega-6 Fatty Acids', 'type' => 'nested', 'category' => 'specialty'],
                    'dha' => ['label' => 'DHA', 'type' => 'nested', 'category' => 'specialty'],
                    'epa' => ['label' => 'EPA', 'type' => 'nested', 'category' => 'specialty']
                ]
            ],

            'cholesterol' => ['label' => 'Cholesterol', 'type' => 'macronutrient', 'category' => 'basic'],
            'sodium' => ['label' => 'Sodium', 'type' => 'macronutrient', 'category' => 'basic'],

            'carbs' => [
                'label' => 'Total Carbohydrate',
                'type' => 'macronutrient',
                'category' => 'basic',
                'required' => true,
                'children' => [
                    'fiber' => ['label' => 'Dietary Fiber', 'type' => 'nested', 'category' => 'basic'],
                    'sugar' => [
                        'label' => 'Total Sugars',
                        'type' => 'nested',
                        'category' => 'basic',
                        'children' => [
                            'added_sugars' => ['label' => 'Includes Added Sugars', 'type' => 'nested', 'category' => 'basic']
                        ]
                    ],
                    'sugar_alcohols' => ['label' => 'Sugar Alcohols', 'type' => 'nested', 'category' => 'detailed'],
                    'starch' => ['label' => 'Starch', 'type' => 'nested', 'category' => 'detailed']
                ]
            ],

            'protein' => [
                'label' => 'Protein',
                'type' => 'macronutrient',
                'category' => 'basic',
                'required' => true,
                'children' => [
                    'histidine' => ['label' => 'Histidine', 'type' => 'nested', 'category' => 'amino_acids'],
                    'isoleucine' => ['label' => 'Isoleucine', 'type' => 'nested', 'category' => 'amino_acids'],
                    'leucine' => ['label' => 'Leucine', 'type' => 'nested', 'category' => 'amino_acids'],
                    'lysine' => ['label' => 'Lysine', 'type' => 'nested', 'category' => 'amino_acids'],
                    'methionine' => ['label' => 'Methionine', 'type' => 'nested', 'category' => 'amino_acids'],
                    'phenylalanine' => ['label' => 'Phenylalanine', 'type' => 'nested', 'category' => 'amino_acids'],
                    'threonine' => ['label' => 'Threonine', 'type' => 'nested', 'category' => 'amino_acids'],
                    'tryptophan' => ['label' => 'Tryptophan', 'type' => 'nested', 'category' => 'amino_acids'],
                    'valine' => ['label' => 'Valine', 'type' => 'nested', 'category' => 'amino_acids']
                ]
            ],

            'salt' => ['label' => 'Salt', 'type' => 'macronutrient', 'category' => 'basic'],
            'alcohol' => ['label' => 'Alcohol', 'type' => 'macronutrient', 'category' => 'specialty'],

            'vitamin_d' => ['label' => 'Vitamin D', 'type' => 'vitamin', 'category' => 'required_fda', 'unit' => 'mcg'],
            'calcium' => ['label' => 'Calcium', 'type' => 'mineral', 'category' => 'required_fda', 'unit' => 'mg'],
            'iron' => ['label' => 'Iron', 'type' => 'mineral', 'category' => 'required_fda', 'unit' => 'mg'],
            'potassium' => ['label' => 'Potassium', 'type' => 'mineral', 'category' => 'required_fda', 'unit' => 'mg'],

            'vitamin_a' => ['label' => 'Vitamin A', 'type' => 'vitamin', 'category' => 'fat_soluble', 'unit' => 'mcg RAE'],
            'vitamin_e' => ['label' => 'Vitamin E', 'type' => 'vitamin', 'category' => 'fat_soluble', 'unit' => 'mg'],
            'vitamin_k' => ['label' => 'Vitamin K', 'type' => 'vitamin', 'category' => 'fat_soluble', 'unit' => 'mcg'],

            'vitamin_c' => ['label' => 'Vitamin C', 'type' => 'vitamin', 'category' => 'water_soluble', 'unit' => 'mg'],
            'thiamin' => ['label' => 'Thiamin (B1)', 'type' => 'vitamin', 'category' => 'b_complex', 'unit' => 'mg'],
            'riboflavin' => ['label' => 'Riboflavin (B2)', 'type' => 'vitamin', 'category' => 'b_complex', 'unit' => 'mg'],
            'niacin' => ['label' => 'Niacin (B3)', 'type' => 'vitamin', 'category' => 'b_complex', 'unit' => 'mg'],
            'pantothenic_acid' => ['label' => 'Pantothenic Acid (B5)', 'type' => 'vitamin', 'category' => 'b_complex', 'unit' => 'mg'],
            'vitamin_b6' => ['label' => 'Vitamin B6', 'type' => 'vitamin', 'category' => 'b_complex', 'unit' => 'mg'],
            'biotin' => ['label' => 'Biotin (B7)', 'type' => 'vitamin', 'category' => 'b_complex', 'unit' => 'mcg'],
            'folate' => ['label' => 'Folate (B9)', 'type' => 'vitamin', 'category' => 'b_complex', 'unit' => 'mcg'],
            'vitamin_b12' => ['label' => 'Vitamin B12', 'type' => 'vitamin', 'category' => 'b_complex', 'unit' => 'mcg'],
            'choline' => ['label' => 'Choline', 'type' => 'vitamin', 'category' => 'water_soluble', 'unit' => 'mg'],

            'phosphorus' => ['label' => 'Phosphorus', 'type' => 'mineral', 'category' => 'major', 'unit' => 'mg'],
            'magnesium' => ['label' => 'Magnesium', 'type' => 'mineral', 'category' => 'major', 'unit' => 'mg'],
            'chloride' => ['label' => 'Chloride', 'type' => 'mineral', 'category' => 'major', 'unit' => 'mg'],

            'iodine' => ['label' => 'Iodine', 'type' => 'mineral', 'category' => 'trace', 'unit' => 'mcg'],
            'zinc' => ['label' => 'Zinc', 'type' => 'mineral', 'category' => 'trace', 'unit' => 'mg'],
            'selenium' => ['label' => 'Selenium', 'type' => 'mineral', 'category' => 'trace', 'unit' => 'mcg'],
            'copper' => ['label' => 'Copper', 'type' => 'mineral', 'category' => 'trace', 'unit' => 'mg'],
            'manganese' => ['label' => 'Manganese', 'type' => 'mineral', 'category' => 'trace', 'unit' => 'mg'],
            'chromium' => ['label' => 'Chromium', 'type' => 'mineral', 'category' => 'trace', 'unit' => 'mcg'],
            'molybdenum' => ['label' => 'Molybdenum', 'type' => 'mineral', 'category' => 'trace', 'unit' => 'mcg'],
            'fluoride' => ['label' => 'Fluoride', 'type' => 'mineral', 'category' => 'trace', 'unit' => 'mg'],

            'caffeine' => ['label' => 'Caffeine', 'type' => 'specialty', 'category' => 'bioactive', 'unit' => 'mg'],
            'beta_carotene' => ['label' => 'Beta-Carotene', 'type' => 'specialty', 'category' => 'carotenoid', 'unit' => 'mcg'],
            'lycopene' => ['label' => 'Lycopene', 'type' => 'specialty', 'category' => 'carotenoid', 'unit' => 'mg'],
            'lutein' => ['label' => 'Lutein', 'type' => 'specialty', 'category' => 'carotenoid', 'unit' => 'mg'],
            'zeaxanthin' => ['label' => 'Zeaxanthin', 'type' => 'specialty', 'category' => 'carotenoid', 'unit' => 'mg'],
            'organic_acids' => ['label' => 'Organic Acids', 'type' => 'specialty', 'category' => 'bioactive', 'unit' => 'g'],

            'serving_size' => ['label' => 'Serving Size', 'type' => 'product_info', 'category' => 'basic'],
            'servings_per_container' => ['label' => 'Servings Per Container', 'type' => 'product_info', 'category' => 'basic'],
            'net_weight' => ['label' => 'Net Weight', 'type' => 'product_info', 'category' => 'basic'],
            'serving_per_package' => ['label' => 'Servings Per Package', 'type' => 'product_info', 'category' => 'basic'],

            'nutri_score' => ['label' => 'Nutri-Score', 'type' => 'rating', 'category' => 'eu_system'],
            'nutri_score_points' => ['label' => 'Nutri-Score Points', 'type' => 'rating', 'category' => 'eu_system'],
            'health_star_rating' => ['label' => 'Health Star Rating', 'type' => 'rating', 'category' => 'au_system'],
            'traffic_light_overall' => ['label' => 'Traffic Light Rating', 'type' => 'rating', 'category' => 'uk_system'],

            'ingredients' => ['label' => 'Ingredients', 'type' => 'text', 'category' => 'basic'],
            'allergens' => ['label' => 'Allergens', 'type' => 'text', 'category' => 'basic'],
            'vitamins' => ['label' => 'Additional Vitamins', 'type' => 'text', 'category' => 'supplementary'],
            'barcode' => ['label' => 'Barcode', 'type' => 'text', 'category' => 'product_info'],
            'preparation_instructions' => ['label' => 'Preparation Instructions', 'type' => 'text', 'category' => 'product_info'],
            'storage_instructions' => ['label' => 'Storage Instructions', 'type' => 'text', 'category' => 'product_info'],
            'country_of_origin' => ['label' => 'Country of Origin', 'type' => 'text', 'category' => 'product_info']
        ];
    }

    public static function get_nutrients_by_category($category = null)
    {
        $hierarchy = self::get_field_hierarchy();

        if ($category === null) {
            $categories = [];
            foreach ($hierarchy as $key => $config) {
                $cat = $config['category'] ?? 'uncategorized';
                $categories[$cat][$key] = $config;
            }
            return $categories;
        }

        $filtered = [];
        foreach ($hierarchy as $key => $config) {
            if (isset($config['category']) && $config['category'] === $category) {
                $filtered[$key] = $config;
            }
        }
        return $filtered;
    }

    public static function get_nutrition_data($product_id)
    {
        $all_meta = get_post_meta($product_id);

        $data = [];
        foreach (self::$fields as $key) {
            $meta_key = 'woonutri_' . $key;
            if (isset($all_meta[$meta_key][0])) {
                $value = $all_meta[$meta_key][0];

                if (is_string($value) && strpos($value, 'a:') === 0) {
                    $value = maybe_unserialize($value);
                }

                if (is_array($value)) {
                    if (!empty($value)) {
                        $data[$key] = $value;
                    }
                } elseif ($value !== '' && $value !== null) {
                    $data[$key] = $value;
                }
            }
        }
        return $data;
    }

    public static function set_nutrition_data($product_id, $data)
    {
        foreach (self::$fields as $key) {
            if (isset($data[$key])) {
                $value = $data[$key];

                if (is_array($value)) {
                    $value = array_map('sanitize_text_field', $value);
                    update_post_meta($product_id, 'woonutri_' . $key, $value);
                } else {
                    update_post_meta($product_id, 'woonutri_' . $key, sanitize_text_field($value));
                }
            }
        }

        self::calculate_and_save_nutri_score($product_id, $data);
    }

    public static function calculate_and_save_nutri_score($product_id, $data = null)
    {
        if (!class_exists('Woonutri_Nutri_Score')) {
            require_once plugin_dir_path(__FILE__) . 'class-nutri-score.php';
        }

        if ($data === null) {
            $data = self::get_nutrition_data($product_id);
        }

        $validation = Woonutri_Nutri_Score::validate_nutrition_data($data);

        if ($validation['is_valid']) {
            $nutri_score_result = Woonutri_Nutri_Score::calculate_nutri_score($data);

            update_post_meta($product_id, 'woonutri_nutri_score', $nutri_score_result['grade']);
            update_post_meta($product_id, 'woonutri_nutri_score_points', $nutri_score_result['score']);
            update_post_meta($product_id, 'woonutri_nutri_score_color', $nutri_score_result['color']);
            update_post_meta($product_id, 'woonutri_nutri_score_details', json_encode($nutri_score_result));

            return $nutri_score_result;
        }

        return false;
    }

    public static function get_nutri_score_data($product_id)
    {
        $grade = get_post_meta($product_id, 'woonutri_nutri_score', true);
        $points = get_post_meta($product_id, 'woonutri_nutri_score_points', true);
        $color = get_post_meta($product_id, 'woonutri_nutri_score_color', true);
        $details_json = get_post_meta($product_id, 'woonutri_nutri_score_details', true);

        if (empty($grade)) {
            return null;
        }

        return [
            'grade' => $grade,
            'points' => intval($points),
            'color' => $color,
            'details' => $details_json ? json_decode($details_json, true) : null
        ];
    }

    public static function get_products_by_nutri_score($grade = null)
    {
        $args = [
            'post_type' => 'product',
            'posts_per_page' => -1,
            'meta_query' => []
        ];

        if ($grade) {
            $args['meta_query'][] = [
                'key' => 'woonutri_nutri_score',
                'value' => $grade,
                'compare' => '='
            ];
        } else {
            $args['meta_query'][] = [
                'key' => 'woonutri_nutri_score',
                'compare' => 'EXISTS'
            ];
        }

        return get_posts($args);
    }

    public static function convert_nutrition_data($data, $serving_size_g, $target_format = 'per_100g')
    {
        if (empty($data) || !$serving_size_g || $serving_size_g <= 0) {
            return $data;
        }

        $converted_data = $data;

        $convertible_fields = [
            'energy_kcal',
            'energy_kj',
            'fat',
            'saturated_fat',
            'monounsaturated_fat',
            'polyunsaturated_fat',
            'trans_fat',
            'cholesterol',
            'sodium',
            'carbs',
            'fiber',
            'sugar',
            'added_sugars',
            'sugar_alcohols',
            'starch',
            'protein',
            'salt',
            'potassium',
            'vitamin_d',
            'calcium',
            'iron',
            'vitamin_a',
            'vitamin_e',
            'vitamin_k',
            'vitamin_c',
            'thiamin',
            'riboflavin',
            'niacin',
            'pantothenic_acid',
            'vitamin_b6',
            'biotin',
            'folate',
            'vitamin_b12',
            'choline',
            'phosphorus',
            'magnesium',
            'chloride',
            'iodine',
            'zinc',
            'selenium',
            'copper',
            'manganese',
            'chromium',
            'molybdenum',
            'fluoride',
            'omega_3',
            'omega_6',
            'dha',
            'epa',
            'caffeine',
            'alcohol',
            'organic_acids',
            'beta_carotene',
            'lycopene',
            'lutein',
            'zeaxanthin'
        ];

        $amino_acids = [
            'histidine',
            'isoleucine',
            'leucine',
            'lysine',
            'methionine',
            'phenylalanine',
            'threonine',
            'tryptophan',
            'valine',
            'alanine',
            'arginine',
            'asparagine',
            'aspartic_acid',
            'cysteine',
            'glutamic_acid',
            'glutamine',
            'glycine',
            'proline',
            'serine',
            'tyrosine'
        ];
        $convertible_fields = array_merge($convertible_fields, $amino_acids);

        if ($target_format === 'per_100g') {
            $multiplier = 100 / $serving_size_g;

            foreach ($convertible_fields as $field) {
                if (isset($data[$field]) && is_numeric($data[$field])) {
                    $converted_data[$field] = round(floatval($data[$field]) * $multiplier, 2);
                }
            }

        } elseif ($target_format === 'per_serving') {
            $multiplier = $serving_size_g / 100;

            foreach ($convertible_fields as $field) {
                if (isset($data[$field]) && is_numeric($data[$field])) {
                    $converted_data[$field] = round(floatval($data[$field]) * $multiplier, 2);
                }
            }
        }

        return $converted_data;
    }

    public static function get_nutrition_data_formatted($product_id, $format = 'per_100g')
    {
        $data = self::get_nutrition_data($product_id);

        if (empty($data)) {
            return $data;
        }

        $serving_size = isset($data['serving_size']) ? $data['serving_size'] : '';
        $serving_size_g = self::extract_serving_size_grams($serving_size);

        if (!$serving_size_g) {
            return $data;
        }

        $current_format = self::detect_data_format($data, $serving_size_g);

        if ($current_format !== $format) {
            $data = self::convert_nutrition_data($data, $serving_size_g, $format);
        }

        return $data;
    }

    public static function extract_serving_size_grams($serving_size)
    {
        if (empty($serving_size)) {
            return null;
        }

        if (preg_match('/(\d+(?:\.\d+)?)\s*g(?:rams?)?/i', $serving_size, $matches)) {
            return floatval($matches[1]);
        }

        if (preg_match('/(\d+(?:\.\d+)?)\s*ml/i', $serving_size, $matches)) {
            return floatval($matches[1]);
        }

        $common_conversions = [
            '/(\d+(?:\.\d+)?)\s*tbsp/i' => function ($matches) {
                return floatval($matches[1]) * 15;
            },
            '/(\d+(?:\.\d+)?)\s*tablespoons?/i' => function ($matches) {
                return floatval($matches[1]) * 15;
            },

            '/(\d+(?:\.\d+)?)\s*tsp/i' => function ($matches) {
                return floatval($matches[1]) * 5;
            },
            '/(\d+(?:\.\d+)?)\s*teaspoons?/i' => function ($matches) {
                return floatval($matches[1]) * 5;
            },

            '/(\d+(?:\.\d+)?)\s*cups?/i' => function ($matches) {
                return floatval($matches[1]) * 240;
            },

            '/(\d+(?:\.\d+)?)\s*fl\.?\s*oz/i' => function ($matches) {
                return floatval($matches[1]) * 30;
            },
            '/(\d+(?:\.\d+)?)\s*oz/i' => function ($matches) {
                return floatval($matches[1]) * 28.35;
            },
        ];

        foreach ($common_conversions as $pattern => $converter) {
            if (preg_match($pattern, $serving_size, $matches)) {
                return $converter($matches);
            }
        }

        if (preg_match('/(\d+(?:\.\d+)?)/', $serving_size, $matches)) {
            $number = floatval($matches[1]);
            if ($number >= 5 && $number <= 500) {
                return $number;
            }
        }

        return null;
    }

    private static function detect_data_format($data, $serving_size_g)
    {
        $energy = isset($data['energy_kcal']) ? floatval($data['energy_kcal']) : 0;

        if ($energy > 0) {
            $energy_per_100g = ($energy / $serving_size_g) * 100;

            if ($energy > 600 || ($energy_per_100g > 50 && $energy_per_100g < 600 && $energy > $energy_per_100g)) {
                return 'per_100g';
            } else {
                return 'per_serving';
            }
        }

        return $serving_size_g <= 200 ? 'per_serving' : 'per_100g';
    }

    public static function get_variation_nutrition_data($variation_id, $parent_id = null)
    {
        if (!$parent_id) {
            $variation = wc_get_product($variation_id);
            if ($variation && $variation->is_type('variation')) {
                $parent_id = $variation->get_parent_id();
            }
        }

        $inherit_from_parent = get_post_meta($variation_id, '_woonutri_inherit_parent', true);

        if ($inherit_from_parent === '' || $inherit_from_parent === 'inherit') {
            $parent_data = $parent_id ? self::get_nutrition_data($parent_id) : [];
            return $parent_data;
        }

        $variation_data = self::get_nutrition_data($variation_id);

        if (empty($variation_data) && $parent_id) {
            return self::get_nutrition_data($parent_id);
        }

        if ($parent_id) {
            $parent_data = self::get_nutrition_data($parent_id);
            foreach ($parent_data as $key => $value) {
                if (!isset($variation_data[$key]) || $variation_data[$key] === '') {
                    $variation_data[$key] = $value;
                }
            }
        }

        return $variation_data;
    }

    public static function set_variation_nutrition_data($variation_id, $data, $mode = 'override')
    {
        update_post_meta($variation_id, '_woonutri_inherit_parent', $mode);

        if ($mode === 'override') {
            self::set_nutrition_data($variation_id, $data);
        }
    }

    public static function has_variation_override($variation_id)
    {
        $inherit = get_post_meta($variation_id, '_woonutri_inherit_parent', true);
        return $inherit === 'override';
    }

    public static function get_display_nutrition_data($product_id)
    {
        $product = wc_get_product($product_id);

        if (!$product) {
            return [];
        }

        if ($product->is_type('variation')) {
            return self::get_variation_nutrition_data($product_id);
        }

        return self::get_nutrition_data($product_id);
    }
}
