<?php
class Woonutri_Nutri_Score {

    public static function calculate_nutri_score( $nutrition_data ) {
        if ( empty( $nutrition_data ) || !is_array( $nutrition_data ) ) {
            return ['score' => 0, 'grade' => 'E', 'color' => '#E63946', 'error' => 'Invalid nutrition data'];
        }

        $per_100g = self::normalize_to_100g( $nutrition_data );

        $negative_points = self::calculate_negative_points( $per_100g );

        $positive_points = self::calculate_positive_points( $per_100g );

        $is_beverage = self::is_beverage( $nutrition_data );

        $final_score = $negative_points - $positive_points;

        $grade = self::score_to_grade( $final_score, $is_beverage );

        $color = self::get_grade_color( $grade );

        return [
            'score' => $final_score,
            'grade' => $grade,
            'color' => $color,
            'negative_points' => $negative_points,
            'positive_points' => $positive_points,
            'is_beverage' => $is_beverage,
            'details' => [
                'energy_points' => self::get_energy_points( $per_100g['energy_kj'] ?? 0 ),
                'saturated_fat_points' => self::get_saturated_fat_points( $per_100g['saturated_fat'] ?? 0 ),
                'sugar_points' => self::get_sugar_points( $per_100g['sugar'] ?? 0, $is_beverage ),
                'sodium_points' => self::get_sodium_points( $per_100g['sodium'] ?? 0 ),
                'fiber_points' => self::get_fiber_points( $per_100g['fiber'] ?? 0 ),
                'protein_points' => self::get_protein_points( $per_100g['protein'] ?? 0 ),
                'fruit_veg_points' => self::get_fruit_vegetable_points( $nutrition_data )
            ]
        ];
    }

    private static function normalize_to_100g( $data ) {
        $serving_size = floatval( $data['serving_size_g'] ?? 100 );

        $per_100g = [];

        $energy_kcal = floatval( $data['energy_kcal'] ?? 0 );
        $energy_kj = floatval( $data['energy_kj'] ?? ( $energy_kcal * 4.184 ) );

        if ( $serving_size != 100 ) {
            $multiplier = 100 / $serving_size;
            $per_100g['energy_kj'] = $energy_kj * $multiplier;
            $per_100g['saturated_fat'] = floatval( $data['saturated_fat'] ?? 0 ) * $multiplier;
            $per_100g['sugar'] = floatval( $data['sugar'] ?? 0 ) * $multiplier;
            $per_100g['sodium'] = floatval( $data['sodium'] ?? 0 ) * $multiplier;
            $per_100g['fiber'] = floatval( $data['fiber'] ?? 0 ) * $multiplier;
            $per_100g['protein'] = floatval( $data['protein'] ?? 0 ) * $multiplier;
        } else {
            $per_100g['energy_kj'] = $energy_kj;
            $per_100g['saturated_fat'] = floatval( $data['saturated_fat'] ?? 0 );
            $per_100g['sugar'] = floatval( $data['sugar'] ?? 0 );
            $per_100g['sodium'] = floatval( $data['sodium'] ?? 0 );
            $per_100g['fiber'] = floatval( $data['fiber'] ?? 0 );
            $per_100g['protein'] = floatval( $data['protein'] ?? 0 );
        }

        if ( $per_100g['sodium'] == 0 && isset( $data['salt'] ) ) {
            $salt_g = floatval( $data['salt'] );
            if ( $serving_size != 100 ) {
                $salt_g *= ( 100 / $serving_size );
            }
            $per_100g['sodium'] = ( $salt_g / 2.5 ) * 1000;
        }

        return $per_100g;
    }

    private static function calculate_negative_points( $per_100g ) {
        $points = 0;

        $points += self::get_energy_points( $per_100g['energy_kj'] );

        $points += self::get_saturated_fat_points( $per_100g['saturated_fat'] );

        $is_beverage = self::is_beverage( $per_100g );
        $points += self::get_sugar_points( $per_100g['sugar'], $is_beverage );

        $points += self::get_sodium_points( $per_100g['sodium'] );

        return $points;
    }

    private static function calculate_positive_points( $per_100g ) {
        $points = 0;

        $points += self::get_fiber_points( $per_100g['fiber'] );

        $points += self::get_protein_points( $per_100g['protein'] );

        $points += self::get_fruit_vegetable_points( $per_100g );

        return $points;
    }

    private static function get_energy_points( $energy_kj ) {
        if ( $energy_kj <= 335 ) return 0;
        if ( $energy_kj <= 670 ) return 1;
        if ( $energy_kj <= 1005 ) return 2;
        if ( $energy_kj <= 1340 ) return 3;
        if ( $energy_kj <= 1675 ) return 4;
        if ( $energy_kj <= 2010 ) return 5;
        if ( $energy_kj <= 2345 ) return 6;
        if ( $energy_kj <= 2680 ) return 7;
        if ( $energy_kj <= 3015 ) return 8;
        if ( $energy_kj <= 3350 ) return 9;
        return 10;
    }

    private static function get_saturated_fat_points( $saturated_fat ) {
        if ( $saturated_fat <= 1 ) return 0;
        if ( $saturated_fat <= 2 ) return 1;
        if ( $saturated_fat <= 3 ) return 2;
        if ( $saturated_fat <= 4 ) return 3;
        if ( $saturated_fat <= 5 ) return 4;
        if ( $saturated_fat <= 6 ) return 5;
        if ( $saturated_fat <= 7 ) return 6;
        if ( $saturated_fat <= 8 ) return 7;
        if ( $saturated_fat <= 9 ) return 8;
        if ( $saturated_fat <= 10 ) return 9;
        return 10;
    }

    private static function get_sugar_points( $sugar, $is_beverage = false ) {
        if ( $is_beverage ) {
            if ( $sugar <= 0 ) return 0;
            if ( $sugar <= 1.5 ) return 1;
            if ( $sugar <= 3 ) return 2;
            if ( $sugar <= 4.5 ) return 3;
            if ( $sugar <= 6 ) return 4;
            if ( $sugar <= 7.5 ) return 5;
            if ( $sugar <= 9 ) return 6;
            if ( $sugar <= 10.5 ) return 7;
            if ( $sugar <= 12 ) return 8;
            if ( $sugar <= 13.5 ) return 9;
            return 10;
        } else {
            if ( $sugar <= 4.5 ) return 0;
            if ( $sugar <= 9 ) return 1;
            if ( $sugar <= 13.5 ) return 2;
            if ( $sugar <= 18 ) return 3;
            if ( $sugar <= 22.5 ) return 4;
            if ( $sugar <= 27 ) return 5;
            if ( $sugar <= 31 ) return 6;
            if ( $sugar <= 36 ) return 7;
            if ( $sugar <= 40 ) return 8;
            if ( $sugar <= 45 ) return 9;
            return 10;
        }
    }

    private static function get_sodium_points( $sodium ) {
        if ( $sodium <= 90 ) return 0;
        if ( $sodium <= 180 ) return 1;
        if ( $sodium <= 270 ) return 2;
        if ( $sodium <= 360 ) return 3;
        if ( $sodium <= 450 ) return 4;
        if ( $sodium <= 540 ) return 5;
        if ( $sodium <= 630 ) return 6;
        if ( $sodium <= 720 ) return 7;
        if ( $sodium <= 810 ) return 8;
        if ( $sodium <= 900 ) return 9;
        return 10;
    }

    private static function get_fiber_points( $fiber ) {
        if ( $fiber <= 0.9 ) return 0;
        if ( $fiber <= 1.9 ) return 1;
        if ( $fiber <= 2.8 ) return 2;
        if ( $fiber <= 3.7 ) return 3;
        if ( $fiber <= 4.7 ) return 4;
        return 5;
    }

    private static function get_protein_points( $protein ) {
        if ( $protein <= 1.6 ) return 0;
        if ( $protein <= 3.2 ) return 1;
        if ( $protein <= 4.8 ) return 2;
        if ( $protein <= 6.4 ) return 3;
        if ( $protein <= 8.0 ) return 4;
        return 5;
    }

    private static function get_fruit_vegetable_points( $data ) {
        $fruit_veg_percent = floatval( $data['fruit_vegetable_percent'] ?? 0 );

        if ( $fruit_veg_percent <= 40 ) return 0;
        if ( $fruit_veg_percent <= 60 ) return 1;
        if ( $fruit_veg_percent <= 80 ) return 2;
        return 5;
    }

    private static function is_beverage( $data ) {
        if ( isset( $data['is_beverage'] ) ) {
            return (bool) $data['is_beverage'];
        }

        $protein = floatval( $data['protein'] ?? 0 );
        $fat = floatval( $data['fat'] ?? 0 );
        $energy_kcal = floatval( $data['energy_kcal'] ?? 0 );

        return ( $protein < 0.8 && $fat < 0.5 && $energy_kcal < 80 );
    }

    private static function score_to_grade( $score, $is_beverage = false ) {
        if ( $is_beverage ) {
            if ( $score <= 1 ) return 'A';
            if ( $score <= 5 ) return 'B';
            if ( $score <= 9 ) return 'C';
            if ( $score <= 13 ) return 'D';
            return 'E';
        } else {
            if ( $score <= -1 ) return 'A';
            if ( $score <= 2 ) return 'B';
            if ( $score <= 10 ) return 'C';
            if ( $score <= 18 ) return 'D';
            return 'E';
        }
    }

    private static function get_grade_color( $grade ) {
        $colors = [
            'A' => '#038141',
            'B' => '#85BB2F',
            'C' => '#FECB02',
            'D' => '#EE8100',
            'E' => '#E63946'
        ];

        return $colors[ $grade ] ?? '#E63946';
    }

    public static function get_score_explanation( $nutrition_data ) {
        $result = self::calculate_nutri_score( $nutrition_data );

        $explanation = [
            'overall' => [
                'grade' => $result['grade'],
                'score' => $result['score'],
                'color' => $result['color'],
                'message' => self::get_grade_message( $result['grade'] )
            ],
            'calculation' => [
                'negative_points' => $result['negative_points'],
                'positive_points' => $result['positive_points'],
                'is_beverage' => $result['is_beverage']
            ],
            'breakdown' => $result['details'] ?? [],
            'recommendations' => self::get_recommendations( $result )
        ];

        return $explanation;
    }

    private static function get_grade_message( $grade ) {
        $messages = [
            'A' => 'Excellent nutritional quality - High in beneficial nutrients, low in nutrients to limit',
            'B' => 'Good nutritional quality - Generally healthy choice with minor limitations',
            'C' => 'Fair nutritional quality - Moderate choice, consume in reasonable quantities',
            'D' => 'Poor nutritional quality - Limited consumption recommended',
            'E' => 'Very poor nutritional quality - Consume occasionally and in small quantities'
        ];

        return $messages[ $grade ] ?? 'Unknown grade';
    }

    private static function get_recommendations( $result ) {
        $recommendations = [];
        $details = $result['details'] ?? [];

        if ( $details['energy_points'] >= 5 ) {
            $recommendations[] = 'Consider reducing portion size due to high energy content';
        }

        if ( $details['saturated_fat_points'] >= 5 ) {
            $recommendations[] = 'High in saturated fat - limit consumption or choose lower-fat alternatives';
        }

        if ( $details['sugar_points'] >= 5 ) {
            $recommendations[] = $result['is_beverage']
                ? 'High sugar content for a beverage - consider sugar-free alternatives'
                : 'High sugar content - consume in moderation';
        }

        if ( $details['sodium_points'] >= 5 ) {
            $recommendations[] = 'High sodium content - may contribute to daily sodium limits';
        }

        if ( $details['fiber_points'] <= 1 ) {
            $recommendations[] = 'Low fiber content - pair with high-fiber foods';
        }

        if ( $details['protein_points'] <= 1 ) {
            $recommendations[] = 'Low protein content - ensure adequate protein from other sources';
        }

        return $recommendations;
    }

    public static function batch_calculate( $products_data ) {
        $results = [];

        foreach ( $products_data as $product_id => $nutrition_data ) {
            $results[ $product_id ] = self::calculate_nutri_score( $nutrition_data );
        }

        return $results;
    }

    public static function validate_nutrition_data( $data ) {
        $required_fields = ['energy_kcal', 'saturated_fat', 'sugar', 'sodium', 'fiber', 'protein'];
        $missing_fields = [];

        foreach ( $required_fields as $field ) {
            if ( !isset( $data[ $field ] ) || $data[ $field ] === '' ) {
                $missing_fields[] = $field;
            }
        }

        return [
            'is_valid' => empty( $missing_fields ),
            'missing_fields' => $missing_fields,
            'message' => empty( $missing_fields )
                ? 'All required fields present for Nutri-Score calculation'
                : 'Missing required fields: ' . implode( ', ', $missing_fields )
        ];
    }
}
