<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * @package WooNutri
 * @since 2.0.0
 */

class Woonutri_Pro_Features
{
    private static $premium_features = [
        'theme_fda' => [
            'title' => 'FDA Theme',
            'description' => 'Professional FDA-compliant nutrition label design',
            'category' => 'themes'
        ],
        'theme_premium' => [
            'title' => 'Premium Theme',
            'description' => 'High-end, premium-looking design with gold accents',
            'category' => 'themes'
        ],
        'theme_dark' => [
            'title' => 'Dark Theme',
            'description' => 'Modern dark theme with light text',
            'category' => 'themes'
        ],
        'theme_us_fda' => [
            'title' => 'US FDA Theme',
            'description' => 'US FDA-compliant nutrition label design',
            'category' => 'themes'
        ],

        'field_vitamins' => [
            'title' => 'Vitamins Field',
            'description' => 'Add detailed vitamins information to nutrition labels',
            'category' => 'fields'
        ],
        'field_potassium' => [
            'title' => 'Potassium Field',
            'description' => 'Add potassium information to nutrition labels',
            'category' => 'fields'
        ],

        'feature_barcode' => [
            'title' => 'Barcode Support',
            'description' => 'Add barcode information to nutrition labels',
            'category' => 'advanced'
        ],
        'feature_expandable' => [
            'title' => 'Expandable Sections',
            'description' => 'Create expandable/collapsible sections in nutrition labels',
            'category' => 'advanced'
        ],

        'feature_daily_values' => [
            'title' => 'Daily Values',
            'description' => 'Show percentage of daily recommended values',
            'category' => 'advanced'
        ],
        'feature_auto_dv' => [
            'title' => 'Auto %DV Calculator',
            'description' => 'Automatically calculate % Daily Values from raw nutrient values',
            'category' => 'advanced'
        ],
        'feature_smart_allergens' => [
            'title' => 'Smart Allergen Bolding',
            'description' => 'Automatically bold allergens in ingredients list',
            'category' => 'advanced'
        ],
    ];

    public static function init()
    {
        add_action('admin_notices', [__CLASS__, 'maybe_show_upgrade_notice']);
        add_filter('woonutri_is_feature_available', [__CLASS__, 'is_feature_available'], 10, 2);

        add_filter('woonutri_nutrition_data', [__CLASS__, 'filter_nutrition_data'], 10, 2);
        add_filter('woonutri_label_options', [__CLASS__, 'filter_label_options'], 10, 2);
    }

    public static function is_feature_available($is_available, $feature_key)
    {
        if ($is_available || !isset(self::$premium_features[$feature_key])) {
            return $is_available;
        }

        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();

        return $is_pro;
    }

    public static function filter_nutrition_data($data, $product_id)
    {
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();

        if (!$is_pro) {
            $premium_fields = ['vitamins', 'barcode'];
            foreach ($premium_fields as $field) {
                if (isset($data[$field])) {
                    unset($data[$field]);
                }
            }
        }

        return $data;
    }

    public static function filter_label_options($options, $args)
    {
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();

        if (!$is_pro) {
            $options['theme'] = 'basic';
            $options['theme_class'] = 'woonutri-theme-basic';
        }

        return $options;
    }

    public static function maybe_show_upgrade_notice()
    {
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'woocommerce_page_woonutri-settings') {
            return;
        }

        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        if ($is_pro) {
            return;
        }

        ?>
        <div class="notice notice-info is-dismissible woonutri-upgrade-notice">
            <h3><?php _e('Upgrade to WooNutri Pro', 'smart-nutrition-labels'); ?></h3>
            <p><?php _e('Get access to premium themes, additional nutrition fields, and advanced features!', 'smart-nutrition-labels'); ?></p>
            <ul class="woonutri-pro-features-list">
                <li><?php _e('Premium label themes (FDA, Premium, Dark, and more)', 'smart-nutrition-labels'); ?></li>
                <li><?php _e('Smart Automation: % Daily Value Calculator & Allergen Bolding', 'smart-nutrition-labels'); ?></li>
                <li><?php _e('Advanced features like expandable sections and barcode support', 'smart-nutrition-labels'); ?></li>
                <li><?php _e('Bulk Editor & CSV Import/Export', 'smart-nutrition-labels'); ?></li>
            </ul>
            <p>
                <a href="<?php echo esc_url(function_exists('woonutri_fs') ? woonutri_fs()->get_upgrade_url() : '#'); ?>"
                    class="button button-primary">
                    <?php _e('Upgrade Now', 'smart-nutrition-labels'); ?>
                </a>
            </p>
        </div>
        <style>
            .woonutri-upgrade-notice {
                padding: 15px;
                border-left-color: #2a7ae2;
            }

            .woonutri-upgrade-notice h3 {
                margin-top: 0;
                color: #2a7ae2;
            }

            .woonutri-pro-features-list {
                list-style: disc;
                padding-left: 20px;
            }

            .woonutri-pro-features-list li {
                margin-bottom: 5px;
            }
        </style>
        <?php
    }

    public static function get_premium_features($category = '')
    {
        if (empty($category)) {
            return self::$premium_features;
        }

        return array_filter(self::$premium_features, function ($feature) use ($category) {
            return $feature['category'] === $category;
        });
    }

    public static function get_feature_details($feature_key)
    {
        return isset(self::$premium_features[$feature_key]) ? self::$premium_features[$feature_key] : null;
    }

    public static function render_upsell_notice($feature_key, $context = 'admin')
    {
        $feature = self::get_feature_details($feature_key);
        if (!$feature) {
            return '';
        }

        ob_start();

        if ($context === 'admin') {
            ?>
            <div class="woonutri-upsell-notice">
                <p>
                    <span class="dashicons dashicons-lock"></span>
                    <?php echo esc_html($feature['title']); ?>             <?php _e('is a premium feature.', 'smart-nutrition-labels'); ?>
                    <a href="<?php echo esc_url(function_exists('woonutri_fs') ? woonutri_fs()->get_upgrade_url() : '#'); ?>">
                        <?php _e('Upgrade to Pro', 'smart-nutrition-labels'); ?>
                    </a>
                </p>
            </div>
            <?php
        } else {
            ?>
            <div class="woonutri-frontend-upsell">
                <p>
                    <?php echo esc_html($feature['title']); ?>             <?php _e('is available in the Pro version.', 'smart-nutrition-labels'); ?>
                </p>
            </div>
            <?php
        }

        return ob_get_clean();
    }
}

Woonutri_Pro_Features::init();
