<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Woonutri_Schema_Output
{

    public static function init()
    {
        add_action('wp_head', [__CLASS__, 'output_nutrition_schema'], 99);
        add_action('woocommerce_structured_data_product', [__CLASS__, 'add_nutrition_to_product_schema'], 10, 2);
    }

    public static function output_nutrition_schema()
    {
        if (!is_product()) {
            return;
        }

        global $product;

        if (!$product || !is_object($product) || !($product instanceof \WC_Product)) {
            $queried_object = get_queried_object();
            if ($queried_object && isset($queried_object->ID)) {
                $product = wc_get_product($queried_object->ID);
            }
        }

        if (!$product || !($product instanceof \WC_Product)) {
            return;
        }

        $product_id = $product->get_id();
        $nutrition_data = self::get_nutrition_data($product_id);

        if (empty($nutrition_data)) {
            return;
        }

        $schema = self::build_nutrition_schema($nutrition_data, $product);

        if (empty($schema)) {
            return;
        }

        echo "\n<!-- WooNutri Structured Data -->\n";
        echo '<script type="application/ld+json">' . "\n";
        echo wp_json_encode($schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        echo "\n</script>\n";
    }

    private static function get_nutrition_data($product_id)
    {
        $profiles = get_post_meta($product_id, '_woonutri_profiles', true);

        if (!empty($profiles) && is_array($profiles) && isset($profiles[0])) {
            return $profiles[0];
        }

        $fields = [
            'energy_kcal',
            'energy_kj',
            'fat',
            'saturated_fat',
            'carbohydrates',
            'sugar',
            'fiber',
            'protein',
            'salt',
            'serving_size',
            'servings_per_container'
        ];

        $data = [];
        foreach ($fields as $field) {
            $value = get_post_meta($product_id, '_woonutri_' . $field, true);
            if ($value !== '') {
                $data[$field] = $value;
            }
        }

        return $data;
    }

    private static function build_nutrition_schema($nutrition_data, $product)
    {
        $nutrition_info = [
            '@type' => 'NutritionInformation',
        ];

        if (!empty($nutrition_data['serving_size'])) {
            $nutrition_info['servingSize'] = $nutrition_data['serving_size'];
        }

        if (!empty($nutrition_data['energy_kcal'])) {
            $nutrition_info['calories'] = $nutrition_data['energy_kcal'] . ' calories';
        }

        if (isset($nutrition_data['fat'])) {
            $nutrition_info['fatContent'] = $nutrition_data['fat'] . ' g';
        }

        if (isset($nutrition_data['saturated_fat'])) {
            $nutrition_info['saturatedFatContent'] = $nutrition_data['saturated_fat'] . ' g';
        }

        if (isset($nutrition_data['trans_fat'])) {
            $nutrition_info['transFatContent'] = $nutrition_data['trans_fat'] . ' g';
        }

        if (isset($nutrition_data['unsaturated_fat'])) {
            $nutrition_info['unsaturatedFatContent'] = $nutrition_data['unsaturated_fat'] . ' g';
        }

        if (isset($nutrition_data['cholesterol'])) {
            $nutrition_info['cholesterolContent'] = $nutrition_data['cholesterol'] . ' mg';
        }

        if (isset($nutrition_data['sodium'])) {
            $nutrition_info['sodiumContent'] = $nutrition_data['sodium'] . ' mg';
        } elseif (isset($nutrition_data['salt'])) {
            $sodium = round(floatval($nutrition_data['salt']) * 1000 * 0.4);
            $nutrition_info['sodiumContent'] = $sodium . ' mg';
        }

        if (isset($nutrition_data['carbohydrates'])) {
            $nutrition_info['carbohydrateContent'] = $nutrition_data['carbohydrates'] . ' g';
        }

        if (isset($nutrition_data['fiber'])) {
            $nutrition_info['fiberContent'] = $nutrition_data['fiber'] . ' g';
        }

        if (isset($nutrition_data['sugar'])) {
            $nutrition_info['sugarContent'] = $nutrition_data['sugar'] . ' g';
        }

        if (isset($nutrition_data['protein'])) {
            $nutrition_info['proteinContent'] = $nutrition_data['protein'] . ' g';
        }

        if (count($nutrition_info) <= 1) {
            return [];
        }

        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'Product',
            'name' => $product->get_name(),
            'description' => wp_strip_all_tags($product->get_short_description()),
            'nutrition' => $nutrition_info,
        ];

        $schema['url'] = get_permalink($product->get_id());

        $image_id = $product->get_image_id();
        if ($image_id) {
            $image_url = wp_get_attachment_url($image_id);
            if ($image_url) {
                $schema['image'] = $image_url;
            }
        }

        $sku = $product->get_sku();
        if ($sku) {
            $schema['sku'] = $sku;
        }

        $brand = get_post_meta($product->get_id(), '_woonutri_brand', true);
        if ($brand) {
            $schema['brand'] = [
                '@type' => 'Brand',
                'name' => $brand,
            ];
        }

        return $schema;
    }

    public static function add_nutrition_to_product_schema($markup, $product)
    {
        $nutrition_data = self::get_nutrition_data($product->get_id());

        if (empty($nutrition_data)) {
            return $markup;
        }

        $nutrition_info = [];

        if (!empty($nutrition_data['serving_size'])) {
            $nutrition_info['servingSize'] = $nutrition_data['serving_size'];
        }

        if (!empty($nutrition_data['energy_kcal'])) {
            $nutrition_info['calories'] = $nutrition_data['energy_kcal'] . ' calories';
        }

        if (isset($nutrition_data['fat'])) {
            $nutrition_info['fatContent'] = $nutrition_data['fat'] . ' g';
        }

        if (isset($nutrition_data['saturated_fat'])) {
            $nutrition_info['saturatedFatContent'] = $nutrition_data['saturated_fat'] . ' g';
        }

        if (isset($nutrition_data['carbohydrates'])) {
            $nutrition_info['carbohydrateContent'] = $nutrition_data['carbohydrates'] . ' g';
        }

        if (isset($nutrition_data['fiber'])) {
            $nutrition_info['fiberContent'] = $nutrition_data['fiber'] . ' g';
        }

        if (isset($nutrition_data['sugar'])) {
            $nutrition_info['sugarContent'] = $nutrition_data['sugar'] . ' g';
        }

        if (isset($nutrition_data['protein'])) {
            $nutrition_info['proteinContent'] = $nutrition_data['protein'] . ' g';
        }

        if (isset($nutrition_data['salt'])) {
            $sodium = round(floatval($nutrition_data['salt']) * 1000 * 0.4);
            $nutrition_info['sodiumContent'] = $sodium . ' mg';
        }

        if (!empty($nutrition_info)) {
            $nutrition_info['@type'] = 'NutritionInformation';
            $markup['nutrition'] = $nutrition_info;
        }

        return $markup;
    }

    public static function test_schema($product_id)
    {
        $product = wc_get_product($product_id);

        if (!$product) {
            return 'Product not found';
        }

        $nutrition_data = self::get_nutrition_data($product_id);

        if (empty($nutrition_data)) {
            return 'No nutrition data found';
        }

        $schema = self::build_nutrition_schema($nutrition_data, $product);

        return wp_json_encode($schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    }
}
